#include "testenv.hpp"
#include "playercore/platform/Platform.hpp"
#include "player/ScopedScheduler.hpp"
#include <condition_variable>
#include <gtest/gtest.h>
#include <mutex>
#include <thread>
#include "test/common/util/gtestenv.hpp"

using namespace twitch;

// Factory methods for creating Scheduler instances
template <class T>
std::shared_ptr<Scheduler> CreateScheduler();

template <>
std::shared_ptr<Scheduler> CreateScheduler<Scheduler>()
{
    auto platform = GTestEnvironment::current->createPlatform();
    return platform->createScheduler("SchedulerTest Scheduler");
}

template <>
std::shared_ptr<Scheduler> CreateScheduler<ScopedScheduler>()
{
    auto platform = GTestEnvironment::current->createPlatform();
    auto scheduler = platform->createScheduler("ScopedSchedulerTest Scheduler");
    return std::make_shared<ScopedScheduler>(scheduler);
}

// Tests for Scheduler implementations
template <class T>
class SchedulerTest : public ::testing::Test {
protected:
    static std::shared_ptr<Scheduler> create()
    {
        return CreateScheduler<T>();
    }
    SchedulerTest()
        : m_scheduler(create())
    {
    }

    void SetUp() override
    {
        ASSERT_TRUE((bool)m_scheduler);
    }

    void TearDown() override
    {
        m_scheduler.reset();
    }

    std::shared_ptr<Scheduler> m_scheduler;
};

// Register Scheduler types
using testing::Types;
typedef Types<Scheduler, ScopedScheduler> SchedulerImplementations;
TYPED_TEST_CASE(SchedulerTest, SchedulerImplementations);

// On PS4, we were able to reproduce the deadlock when adding a 1 second sleep
// before setting m_run to false in its destructor (see CVP-1384)
TYPED_TEST(SchedulerTest, CreateAndDestroyScheduler)
{
    static const size_t numRuns = 5;
    for (size_t i = 0; i < numRuns; i++) {
        auto scheduler = this->create();
        scheduler.reset();
    }
}

TYPED_TEST(SchedulerTest, ScheduleTest)
{
    std::mutex mutex;
    std::condition_variable condition;
    bool done = false;

    this->m_scheduler->schedule([&]() {
        {
            std::unique_lock<std::mutex> lock(mutex);
            done = true;
        }
        condition.notify_one();
    });

    {
        std::unique_lock<std::mutex> lock(mutex);
        EXPECT_TRUE(condition.wait_for(lock, std::chrono::seconds(5), [&done]() { return done; }));
    }

    // the task should have ran
    EXPECT_TRUE(done);
}

TYPED_TEST(SchedulerTest, ScheduleAndWaitTest)
{
    bool done = false;
    this->m_scheduler->scheduleAndWait([&]() {
        done = true;
    });

    // the task should have ran
    EXPECT_TRUE(done);
}

TYPED_TEST(SchedulerTest, ScheduleAndWaitInSchedule)
{
    std::mutex mutex;
    std::condition_variable condition;
    bool firstTask = false;
    bool secondTask = false;

    this->m_scheduler->schedule([&] {
        {
            std::unique_lock<std::mutex> lock(mutex);
            firstTask = true;
        }

        this->m_scheduler->scheduleAndWait([&]() {
            {
                std::unique_lock<std::mutex> lock(mutex);
                secondTask = true;
            }

            condition.notify_one();
        });
    });

    {
        std::unique_lock<std::mutex> lock(mutex);
        EXPECT_TRUE(condition.wait_for(lock, std::chrono::seconds(5), [&firstTask, &secondTask]() { return secondTask && firstTask; }));
    }

    EXPECT_TRUE(firstTask);
    EXPECT_TRUE(secondTask);
}

TYPED_TEST(SchedulerTest, DestroyWaitsForTaskToBeCompleted)
{
    std::mutex mutex;
    std::condition_variable condition;
    bool done = false;

    {
        auto scheduler = this->create();
        scheduler->schedule([&]() {
            std::this_thread::sleep_for(std::chrono::milliseconds(1000));
            {
                std::unique_lock<std::mutex> lock(mutex);
                done = true;
            }
            condition.notify_one();
        });
        std::this_thread::sleep_for(std::chrono::milliseconds(200));
    }

    EXPECT_TRUE(done);
}

TYPED_TEST(SchedulerTest, CancelBeforeScheduleCanRun)
{
    bool done = false;

    auto cancelToken = this->m_scheduler->schedule([&]() {
        done = true;
    },
        std::chrono::milliseconds(250));

    cancelToken->cancel();
    std::this_thread::sleep_for(std::chrono::milliseconds(500)); // give enough time for the task to run if it wasn't canceled properly

    EXPECT_FALSE(done);
}

TYPED_TEST(SchedulerTest, NestedTasks)
{
    bool done = false;
    std::mutex mutex;
    std::condition_variable condition;

    this->m_scheduler->schedule([&]() {
        this->m_scheduler->schedule([&]() {
            {
                std::unique_lock<std::mutex> lock(mutex);
                done = true;
            }
        });
    });

    std::unique_lock<std::mutex> lock(mutex);
    EXPECT_TRUE(condition.wait_for(lock, std::chrono::seconds(1), [&done]() { return done; }));
    EXPECT_TRUE(done);
}

TYPED_TEST(SchedulerTest, CancelMiddleTask)
{
    bool done[3] = { false, false, false };
    std::mutex mutex;
    std::condition_variable condition;

    this->m_scheduler->schedule([&]() {
        {
            std::unique_lock<std::mutex> lock(mutex);
            done[0] = true;
        }
    });

    auto cancelToken = this->m_scheduler->schedule([&]() {
        {
            std::unique_lock<std::mutex> lock(mutex);
            done[1] = true;
        }
    },
        std::chrono::milliseconds(250));

    this->m_scheduler->schedule([&]() {
        {
            std::unique_lock<std::mutex> lock(mutex);
            done[2] = true;
        }
        condition.notify_one();
    });

    cancelToken->cancel();

    {
        std::unique_lock<std::mutex> lock(mutex);
        EXPECT_TRUE(condition.wait_for(lock, std::chrono::seconds(1), [&]() { return done[2]; }));
    }

    EXPECT_TRUE(done[0]);
    EXPECT_FALSE(done[1]);
}

TYPED_TEST(SchedulerTest, Ordering)
{
    bool done[2] = { false, false };
    std::mutex mutex;
    std::condition_variable condition;

    // Schedule a task to be run later
    this->m_scheduler->schedule([&]() {
        {
            std::unique_lock<std::mutex> lock(mutex);
            done[0] = true;
        }
        condition.notify_one();
    },
        std::chrono::milliseconds(250));

    // Then schedule one to be run immediately
    this->m_scheduler->schedule([&]() {
        {
            std::unique_lock<std::mutex> lock(mutex);
            done[1] = true;
        }
        condition.notify_one();
    });

    // The second task should run first
    {
        std::unique_lock<std::mutex> lock(mutex);
        EXPECT_TRUE(condition.wait_for(lock, std::chrono::seconds(1), [&]() { return done[1] && !done[0]; }));
    }

    // Then the first task should run
    {
        std::unique_lock<std::mutex> lock(mutex);
        EXPECT_TRUE(condition.wait_for(lock, std::chrono::seconds(1), [&]() { return done[0]; }));
    }
}

// This tests creating scheduling a new task, within a task, while the ScopedScheduler is being destroyed.
TYPED_TEST(SchedulerTest, SpawnDuringScopedDTor)
{
    std::mutex mutex;
    std::condition_variable condition;

    {
        auto scheduler = std::make_shared<twitch::ScopedScheduler>(this->create());
        bool outerRunning = false;

        std::shared_ptr<Cancellable> outerTask = scheduler->schedule([&]()
        {
            // This task will wait for cancellation, then it will spawn a new task !
            std::cerr << "Outer task running" << std::endl;
            {
                std::unique_lock<std::mutex> lock(mutex);
                outerRunning = true;
            }

            // now wait until somebody cancels you !
            // Busy loop is needed because canceling a task will not wake up a condition_variable
            while (true) {
                std::this_thread::sleep_for(std::chrono::milliseconds(100));
                {
                    std::unique_lock<std::mutex> lock(mutex);
                    if (!outerRunning) {
                        break;
                    }
                }
            }

        });

        std::cerr << "Test is waiting for outer task to run" << std::endl;
        {
            std::unique_lock<std::mutex> lock(mutex);
            EXPECT_TRUE(condition.wait_for(lock, std::chrono::seconds(1), [&]() { return outerRunning; }));
            outerRunning = false;
        }

        scheduler.reset();
        // outer is running now !
        // there's nothing else to do, let ScopedScheduler go out of scope, OuterTask will detect that
        // then OuterTask will attempt to schedule a new task, which should fail, or should be done before the ScopedScheduler goes out
        std::cerr << "Test will destroy the ScopedScheduler (goes out of scope)" << std::endl;
    }
    condition.notify_one(); // to let the outer task know we are done canceling
    std::cerr << "Test notifies outer task that it has been cancelled" << std::endl;

    condition.notify_one();

    // give some time for the 'innerTask' to run in case it does get scheduled and thus fail the test.
    std::this_thread::sleep_for(std::chrono::seconds(2));
}
