#pragma once
#include "media/sourceformat.hpp"
#include "playercore/nativeplatform.hpp"
#include <functional>
#include <gtest/gtest.h>
#include <utility>
#include "test/common/util/gtestenv.hpp"

using namespace twitch;
using namespace twitch::media;
/**
 * MediaPlayerAudioDecoderTestBase test the AudioDecoder framework
 */
class MediaPlayerAudioDecoderTestBase : public ::testing::Test {
    void SetUp() override
    {
        m_sourceFormat = std::make_shared<SourceFormat>(MediaType::Audio_AAC, "audio/aac");
        m_audioDecoder = createDecoder();
        ASSERT_TRUE(m_audioDecoder != nullptr) << "Failed to construct AudioDecoder";
    }

    void TearDown() override
    {
        m_audioDecoder.reset();
    }

protected:
    std::shared_ptr<MediaDecoder> createDecoder()
    {
        auto platform = GTestEnvironment::current->createPlatform();
        auto nativePlatform = std::static_pointer_cast<NativePlatform>(platform);
        return nativePlatform->createDecoder(m_sourceFormat);
    }

    const int InputSampleSize = 16;
    const int InputSampleRate = 44100;

#if PLAYERCORE_OS_PS4
    const int ExpectedOutputSampleSize = InputSampleSize;
    // PS4 only support 48khz output rate and resampling is done in the decode process
    const int ExpectedOutputSampleRate = 48000;
#else
    const int ExpectedOutputSampleSize = InputSampleSize;
    const int ExpectedOutputSampleRate = InputSampleRate;
#endif

    std::shared_ptr<SourceFormat> m_sourceFormat;
    std::shared_ptr<MediaDecoder> m_audioDecoder;
};

TEST_F(MediaPlayerAudioDecoderTestBase, AudioDecoderConstruct)
{
    auto sf = std::make_shared<SourceFormat>(MediaType::Audio_AAC, "audio/aac");
    std::shared_ptr<MediaDecoder> ad = createDecoder();
    ASSERT_TRUE(ad != nullptr) << "Failed to construct AudioDecoder";
}

TEST_F(MediaPlayerAudioDecoderTestBase, AudioDecoderInvalidArguments)
{
    SourceFormat output(MediaType::Audio_PCM, "audio/pcm");
    ASSERT_EQ(MediaResult::ErrorInvalidParameter, m_audioDecoder->configure(*m_sourceFormat, output));
}

TEST_F(MediaPlayerAudioDecoderTestBase, AudioDecoderValidA)
{
    const int numChannels = 1;
    SourceFormat input(MediaType::Audio_AAC, "audio/aac");
    input.setInt(MediaFormat::Audio_SampleSize, InputSampleSize);
    input.setInt(MediaFormat::Audio_SampleRate, InputSampleRate);
    input.setInt(MediaFormat::Audio_ChannelCount, numChannels);
    input.setCodecData(MediaFormat::Audio_AAC_ESDS, { '\x12', '\xb' });

    SourceFormat output(MediaType::Audio_PCM, "audio/pcm");

    MediaResult mr = m_audioDecoder->configure(input, output);
    ASSERT_EQ(MediaResult::Ok, mr);

    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_SampleSize));
    EXPECT_EQ(ExpectedOutputSampleSize, output.getInt(MediaFormat::Audio_SampleSize));
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_SampleRate));
    EXPECT_EQ(ExpectedOutputSampleRate, output.getInt(MediaFormat::Audio_SampleRate));
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_ChannelCount));
    EXPECT_EQ(numChannels, output.getInt(MediaFormat::Audio_ChannelCount));
}

TEST_F(MediaPlayerAudioDecoderTestBase, AudioDecoderValidB)
{
    const int numChannels = 2;

    SourceFormat input(MediaType::Audio_AAC, "audio/aac");
    input.setInt(MediaFormat::Audio_SampleSize, InputSampleSize);
    input.setInt(MediaFormat::Audio_SampleRate, InputSampleRate);
    input.setInt(MediaFormat::Audio_ChannelCount, numChannels);
    input.setCodecData(MediaFormat::Audio_AAC_ESDS, { '\x12', '\x10' });

    SourceFormat output(MediaType::Audio_PCM, "audio/pcm");

    MediaResult mr = m_audioDecoder->configure(input, output);
    ASSERT_EQ(MediaResult::Ok, mr);

    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_SampleSize));
    EXPECT_EQ(ExpectedOutputSampleSize, output.getInt(MediaFormat::Audio_SampleSize));
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_SampleRate));
    EXPECT_EQ(ExpectedOutputSampleRate, output.getInt(MediaFormat::Audio_SampleRate));
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_ChannelCount));
    EXPECT_EQ(numChannels, output.getInt(MediaFormat::Audio_ChannelCount));
}

TEST_F(MediaPlayerAudioDecoderTestBase, AudioDecoderSame)
{
    const int numChannels = 2;
    SourceFormat input(MediaType::Audio_AAC, "audio/aac");
    input.setInt(MediaFormat::Audio_SampleSize, InputSampleSize);
    input.setInt(MediaFormat::Audio_SampleRate, InputSampleRate);
    input.setInt(MediaFormat::Audio_ChannelCount, numChannels);
    input.setCodecData(MediaFormat::Audio_AAC_ESDS, { '\x12', '\x10' });

    SourceFormat output(MediaType::Audio_PCM, "audio/pcm");

    MediaResult mr = m_audioDecoder->configure(input, output);
    ASSERT_EQ(MediaResult::Ok, mr);
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_SampleSize));
    EXPECT_EQ(ExpectedOutputSampleSize, output.getInt(MediaFormat::Audio_SampleSize));
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_SampleRate));
    EXPECT_EQ(ExpectedOutputSampleRate, output.getInt(MediaFormat::Audio_SampleRate));
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_ChannelCount));
    EXPECT_EQ(numChannels, output.getInt(MediaFormat::Audio_ChannelCount));

    mr = m_audioDecoder->configure(input, output);
    ASSERT_EQ(MediaResult::Ok, mr);
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_SampleSize));
    EXPECT_EQ(ExpectedOutputSampleSize, output.getInt(MediaFormat::Audio_SampleSize));
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_SampleRate));
    EXPECT_EQ(ExpectedOutputSampleRate, output.getInt(MediaFormat::Audio_SampleRate));
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_ChannelCount));
    EXPECT_EQ(numChannels, output.getInt(MediaFormat::Audio_ChannelCount));
}

TEST_F(MediaPlayerAudioDecoderTestBase, AudioDecoderReconfigure)
{
    SourceFormat input(MediaType::Audio_AAC, "audio/aac");
    input.setInt(MediaFormat::Audio_SampleSize, InputSampleSize);
    input.setInt(MediaFormat::Audio_SampleRate, InputSampleRate);
    input.setInt(MediaFormat::Audio_ChannelCount, 2);
    input.setCodecData(MediaFormat::Audio_AAC_ESDS, { '\x12', '\x10' });

    SourceFormat output(MediaType::Audio_PCM, "audio/pcm");

    MediaResult mr = m_audioDecoder->configure(input, output);
    ASSERT_EQ(MediaResult::Ok, mr);
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_SampleSize));
    EXPECT_EQ(ExpectedOutputSampleSize, output.getInt(MediaFormat::Audio_SampleSize));
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_SampleRate));
    EXPECT_EQ(ExpectedOutputSampleRate, output.getInt(MediaFormat::Audio_SampleRate));
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_ChannelCount));
    EXPECT_EQ(2, output.getInt(MediaFormat::Audio_ChannelCount));

    input.setInt(MediaFormat::Audio_ChannelCount, 1);
    input.setCodecData(MediaFormat::Audio_AAC_ESDS, { '\x12', '\xb' });

    mr = m_audioDecoder->configure(input, output);
    ASSERT_EQ(MediaResult::Ok, mr);
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_SampleSize));
    EXPECT_EQ(ExpectedOutputSampleSize, output.getInt(MediaFormat::Audio_SampleSize));
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_SampleRate));
    EXPECT_EQ(ExpectedOutputSampleRate, output.getInt(MediaFormat::Audio_SampleRate));
    EXPECT_TRUE(output.hasInt(MediaFormat::Audio_ChannelCount));
    EXPECT_EQ(1, output.getInt(MediaFormat::Audio_ChannelCount));
}
