#pragma once
#include "playercore/format.hpp"
#include "playercore/nativeplatform.hpp"
#include "player/MediaPlayer.hpp"
#include "media/sourceformat.hpp"
#include "state_change_monitor.hpp"
#include <functional>
#include <gtest/gtest.h>
#include <utility>
#include "test/common/util/gtestenv.hpp"

using namespace twitch;
using namespace twitch::media;
/**
 * MediaPlayerVideoDecoderTestBase test the VideoDecoder framework
 */
class MediaPlayerVideoDecoderTestBase : public ::testing::Test {
    void SetUp() override
    {
        m_sourceFormat = std::make_shared<SourceFormat>(MediaType::Video_AVC, "video/avc");
        m_videoDecoder = createDecoder();
        ASSERT_TRUE(m_videoDecoder != nullptr) << "Failed to construct VideoDecoder";
    }

    void TearDown() override
    {
        m_videoDecoder.reset();
    }

protected:
    std::shared_ptr<MediaDecoder> createDecoder()
    {
        auto platform = GTestEnvironment::current->createPlatform();
        auto nativePlatform = std::static_pointer_cast<NativePlatform>(platform);
        return nativePlatform->createDecoder(m_sourceFormat);
    }

    std::shared_ptr<SourceFormat> m_sourceFormat;
    std::shared_ptr<MediaDecoder> m_videoDecoder;
};

// PS4 doesn't care about the parameters you pass to configure. It always configure itself at the highest resolution as of this writing
#if !PLAYERCORE_OS_PS4
TEST_F(MediaPlayerVideoDecoderTestBase, VideoDecoderInvalidConfigure)
{
    auto input = std::make_shared<SourceFormat>(MediaType::Audio_AAC, "crap/funny");
    SourceFormat output;
    MediaResult mr = m_videoDecoder->configure(*input, output);
    ASSERT_EQ(MediaResult::ErrorInvalidParameter, mr);
}
#endif

TEST_F(MediaPlayerVideoDecoderTestBase, VideoDecoderConfigure)
{
    static const int videoWidth = 1088;
    static const int videoHeight = 800;

    auto sourceFormat = SourceFormat::createVideoFormat(MediaType::Video_AVC, videoWidth, videoHeight);
    sourceFormat->setInt(MediaFormat::Video_AVC_Level, 52);
    sourceFormat->setInt(MediaFormat::Video_AVC_Profile, 100);

    SourceFormat output;
    MediaResult mr = m_videoDecoder->configure(*sourceFormat, output);
    ASSERT_EQ(MediaResult::Ok, mr);

    // The PS4 decoder might output to a bigger sized texture. This is required for ABS to work. It shouldn't output to a smaller sized texture however.
    ASSERT_TRUE(output.hasInt(MediaFormat::Video_Width));
    EXPECT_LE(videoWidth, output.getInt(MediaFormat::Video_Width));
    ASSERT_TRUE(output.hasInt(MediaFormat::Video_Height));
    EXPECT_LE(videoHeight, output.getInt(MediaFormat::Video_Height));
}

TEST_F(MediaPlayerVideoDecoderTestBase, VideoDecoderStressCtorDtor)
{
    for (int i = 0; i < 10; ++i) {
        m_videoDecoder.reset();
        m_videoDecoder = createDecoder();
        ASSERT_TRUE(m_videoDecoder);

        static const int videoWidth = 1088;
        static const int videoHeight = 800;

        auto sourceFormat = SourceFormat::createVideoFormat(MediaType::Video_AVC, videoWidth, videoHeight);
        sourceFormat->setInt(MediaFormat::Video_AVC_Level, 52);
        sourceFormat->setInt(MediaFormat::Video_AVC_Profile, 100);

        SourceFormat output;
        MediaResult mr = m_videoDecoder->configure(*sourceFormat, output);
        ASSERT_EQ(MediaResult::Ok, mr);

        ASSERT_TRUE(output.hasInt(MediaFormat::Video_Width));
        EXPECT_LE(videoWidth, output.getInt(MediaFormat::Video_Width));
        ASSERT_TRUE(output.hasInt(MediaFormat::Video_Height));
        EXPECT_LE(videoHeight, output.getInt(MediaFormat::Video_Height));
    }
}
