#include "event.hpp"
#include "event_listener.hpp"
#include <cstring>
#include <gtest/gtest.h>
#include <list>
#include <tuple>

using namespace twitch::test;
using namespace ::testing;
using namespace ::testing::internal;

TEST(EventTest, ReceiveWhenSubscribed)
{
    Event<void(int)> event;
    EventListener<int> listener;

    EXPECT_TRUE(listener.subscribe(&event)) << "Could not subscribe to Event<void(int)>";
    EXPECT_EQ(0U, listener.getEvents().size()) << "Received event before subscribing";

    event(1);
    EXPECT_EQ(1U, listener.getEvents().size()) << "Did not receive an event when subscribed";
}

TEST(EventTest, NoEventsWhenUnubscribed)
{
    Event<void(int)> event;
    EventListener<int> listener;

    event(1);
    EXPECT_EQ(0U, listener.getEvents().size()) << "Received event before subscribing";

    EXPECT_TRUE(listener.subscribe(&event)) << "Could not subscribe to Event<void(int)>";
    EXPECT_TRUE(listener.unsubscribe()) << "Could not unsubscribe from Event<void(int)>";

    event(1);
    EXPECT_EQ(0U, listener.getEvents().size()) << "Received event after unsubscribe";
}

TEST(EventTest, LoopIntEvents)
{
    Event<void(int)> event;
    EventListener<int> listener;

    ASSERT_TRUE(listener.subscribe(&event)) << "Could not subscribe to Event<void(int)>";

    const size_t numEvents = 10;
    for (size_t i = 0; i < numEvents; i++) {
        int intEvent = static_cast<int>(i);
        event(intEvent);

        const auto receivedEvents = listener.getEvents();
        EXPECT_EQ(i + 1, receivedEvents.size());
        EXPECT_EQ(intEvent, std::get<0>(receivedEvents.back()));
    }

    const auto receivedEvents = listener.getEvents();
    EXPECT_EQ(numEvents, receivedEvents.size());
    EXPECT_EQ(static_cast<int>(numEvents - 1), std::get<0>(receivedEvents.back()));
}

TEST(EventTest, Multiplelisteners)
{
    Event<void(int)> event;

    EventListener<int> listener1;
    EventListener<int> listener2;

    EXPECT_TRUE(listener1.subscribe(&event)) << "Could not subscribe to Event<void(int)>";
    EXPECT_TRUE(listener2.subscribe(&event)) << "Could not subscribe to Event<void(int)>";

    std::list<std::tuple<int>> firedEvents;

    const int numEvents = 10;
    for (int i = 0; i < numEvents; i++) {
        event(i);
        firedEvents.emplace_back(i);
    }

    EXPECT_EQ(firedEvents, listener1.getEvents());
    EXPECT_EQ(firedEvents, listener2.getEvents());
}

TEST(EventTest, UnsubscribeAll)
{
    Event<void(int)> event;

    EventListener<int> listener1;
    EventListener<int> listener2;

    EXPECT_TRUE(listener1.subscribe(&event)) << "Could not subscribe to Event<void(int)>";
    EXPECT_TRUE(listener2.subscribe(&event)) << "Could not subscribe to Event<void(int)>";

    event.unsubscribeAll();

    event(1);

    EXPECT_EQ(0U, listener1.getEvents().size()) << "Received event after unsubscribe all";
    EXPECT_EQ(0U, listener2.getEvents().size()) << "Received event after unsubscribe all";
}
