#include "Log.hpp"
#include "player/MediaPlayer.hpp"
#include "sink/ThreadScheduler.hpp"
#include "../mock/mock_media_player.hpp"
#include "../mock/mock_media_source.hpp"
#include "../mock/mock_platform.hpp"
#include "player_error_monitor.hpp"
#include "player_listener.hpp"
#include "EmptyPlayerListener.hpp"
#include "state_change_monitor.hpp"
#include "debug/PrintLog.hpp"
#include <gtest/gtest.h>
#include <memory>
#include <string>
#include <vector>

using namespace twitch;
using namespace twitch::test;
using namespace ::testing;
using namespace ::testing::internal;
using namespace std::placeholders;

class MediaPlayerTest : public ::testing::Test, public EmptyPlayerListener {
protected:
    std::shared_ptr<PrintLog> m_log;
    std::shared_ptr<NiceMock<MockPlatform>> m_platform;

    std::shared_ptr<MediaPlayer> m_player;

    std::string m_path = "url";
    const std::string m_platformName = "Mock";
    const std::map<std::string, std::string> m_emptyAnalyticsProperties{};

    PlayerErrorMonitor m_errorMonitor;
    std::unique_ptr<PlayerStateChangeMonitor> m_stateMonitor;
    twitch::Capabilities m_capabilities;

    void SetUp()
    {
        m_log = std::make_shared<PrintLog>();
        m_platform = std::make_shared<NiceMock<MockPlatform>>();

        ON_CALL(*m_platform, getLog())
            .WillByDefault(Return(m_log));

        ON_CALL(*m_platform, getName())
            .WillByDefault(ReturnRef(m_platformName));

        ON_CALL(*m_platform, createScheduler(_))
            .WillByDefault(Invoke([this](const std::string& name) {
                return std::make_shared<ThreadScheduler>(*m_platform, name);
            }));

        ON_CALL(*m_platform, getCapabilities())
            .WillByDefault(ReturnRef(m_capabilities));

        ON_CALL(*m_platform, getAnalyticsProperties())
            .WillByDefault(ReturnRef(m_emptyAnalyticsProperties));

        m_player = std::make_shared<MediaPlayer>(*this, m_platform);
        m_stateMonitor = std::unique_ptr<PlayerStateChangeMonitor>(new PlayerStateChangeMonitor(m_player->getState()));
    }

    void TearDown()
    {
        m_player.reset();
        m_stateMonitor.reset();
        m_platform.reset();
    }

    void onError(const twitch::Error& error) override
    {
        m_errorMonitor.onError(error);
    }

    void onStateChanged(twitch::Player::State state) override
    {
        m_stateMonitor->onStateChanged(state);
    }

    void onSessionData(const std::map<std::string, std::string>& properties) override
    {
        (void)properties;
    }

};

TEST_F(MediaPlayerTest, ConstructorTrivial)
{
    PlayerEventListener listener;
    MediaPlayer player(listener, m_platform);
    EXPECT_EQ(Player::State::Idle, player.getState());
}

TEST_F(MediaPlayerTest, ConstructMultipleInstances)
{
    PlayerEventListener listener;
    MediaPlayer player1(listener, m_platform);
    MediaPlayer player2(listener, m_platform);

    EXPECT_EQ(Player::State::Idle, player1.getState());
    EXPECT_EQ(Player::State::Idle, player2.getState());
}

TEST_F(MediaPlayerTest, LoadEmptyURL)
{
    m_player->load("");
    EXPECT_EQ(Player::State::Idle, m_player->getState());
}

TEST_F(MediaPlayerTest, LoadNullSource)
{
    ON_CALL(*m_platform, createSource(_, _, _, _))
        .WillByDefault(Invoke([](const std::string&, const MediaType&, MediaSource::Listener&, std::shared_ptr<Scheduler>) {
            return std::unique_ptr<MediaSource>();
        }));

    m_errorMonitor.waitFor(twitch::ErrorSource::Source, twitch::MediaResult::ErrorNoSource, "Source create failed", [this]() {
        m_player->load(m_path);
    },
        std::chrono::seconds(5));
}

TEST_F(MediaPlayerTest, LoadInvalidSource)
{
    auto mediaSource = new NiceMock<MockMediaSource>();
    ON_CALL(*mediaSource, open())
        .WillByDefault(Return());

    ON_CALL(*m_platform, createSource(_, _, _, _))
        .WillByDefault(Invoke([mediaSource](const std::string&, const MediaType&, MediaSource::Listener&, std::shared_ptr<Scheduler>) {
            return std::unique_ptr<MediaSource>(mediaSource);
        }));

    m_errorMonitor.waitFor(twitch::ErrorSource::Source, twitch::MediaResult::Error, "Source open failed", [this]() {
        m_player->load(m_path);
    },
        std::chrono::seconds(5));

    EXPECT_EQ(Player::State::Idle, m_player->getState());
}

TEST_F(MediaPlayerTest, LoadGoodSource)
{
    MediaSource::Listener* sourceListener = nullptr;
    auto mediaSource = new NiceMock<MockMediaSource>();
    ON_CALL(*mediaSource, open())
        .WillByDefault(Return());

    std::vector<twitch::Quality> qualities;
    ON_CALL(*mediaSource, getQualities())
        .WillByDefault(ReturnRef(qualities));

    ON_CALL(*mediaSource, open())
        .WillByDefault(Invoke([&]() {
            sourceListener->onSourceOpened();
        }));

    std::map<std::string, std::string> properties;
    ON_CALL(*mediaSource, getProperties())
        .WillByDefault(ReturnRef(properties));

    ON_CALL(*m_platform, createSource(_, _, _, _))
        .WillByDefault(Invoke([&](const std::string&, const MediaType&, MediaSource::Listener& listener, std::shared_ptr<Scheduler>) {
            sourceListener = &listener;
            return std::unique_ptr<MediaSource>(mediaSource);
        }));

    EXPECT_TRUE(m_stateMonitor->waitFor(Player::State::Ready, [&]() {
        m_player->load(m_path);
    },
        std::chrono::seconds(5)));

    EXPECT_EQ(0U, m_errorMonitor.getErrors().size());
}

TEST_F(MediaPlayerTest, StartNullSource)
{
    ON_CALL(*m_platform, createSource(_, _, _, _))
        .WillByDefault(Invoke([](const std::string&, const MediaType&, MediaSource::Listener&, std::shared_ptr<Scheduler>) {
            return std::unique_ptr<MediaSource>();
        }));

    m_errorMonitor.waitFor(twitch::ErrorSource::Source, twitch::MediaResult::ErrorNoSource, "Source create failed", [this]() {
        m_player->load(m_path);
    },
        std::chrono::seconds(5));
}

TEST_F(MediaPlayerTest, StartInvalidSource)
{
    auto mediaSource = new NiceMock<MockMediaSource>();
    ON_CALL(*mediaSource, open())
        .WillByDefault(Return());

    ON_CALL(*mediaSource, read(_))
        .WillByDefault(Return());

    ON_CALL(*m_platform, createSource(_, _, _, _))
        .WillByDefault(Invoke([mediaSource](const std::string&, const MediaType&, MediaSource::Listener&, std::shared_ptr<Scheduler>) {
            return std::unique_ptr<MediaSource>(mediaSource);
        }));

    m_errorMonitor.waitFor(twitch::ErrorSource::Source, twitch::MediaResult::Error, "Source open failed", [this]() {
        m_player->load(m_path);
    },
        std::chrono::seconds(5));

    m_errorMonitor.waitFor(twitch::ErrorSource::Source, twitch::MediaResult::Error, "Source open failed", [this]() {
        m_player->play();
    },
        std::chrono::seconds(5));
}

TEST_F(MediaPlayerTest, StartValidSource)
{
    auto mediaSource = new NiceMock<MockMediaSource>();
    ON_CALL(*mediaSource, open())
        .WillByDefault(Return());

    ON_CALL(*m_platform, createSource(_, _, _, _))
        .WillByDefault(Invoke([mediaSource](const std::string&, const MediaType&, MediaSource::Listener&, std::shared_ptr<Scheduler>) {
            return std::unique_ptr<MediaSource>(mediaSource);
        }));

    m_player->load(m_path);
    m_player->play();
    EXPECT_EQ(0U, m_errorMonitor.getErrors().size());
}
