package main

import (
	"context"
	"fmt"
	"io/ioutil"
	"os"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/safety/aegis/internal/auth"
	"code.justin.tv/safety/aegis/internal/clients"

	"github.com/ghodss/yaml"
	"github.com/pkg/errors"
)

var (
	// ErrNoConfigFound is returned when an environment we don't have a default
	// config for is used.
	ErrNoConfigFound = errors.New("no default config found")
)

type appConfig struct {
	Auth auth.Config
	// Datastore datastore.Config
	Clients clients.Config
}

var (
	defaultConfigs = map[string]appConfig{
		"development": {
			Auth: auth.Config{
				GuardianClientID: "0ce90ad2-db54-4f82-b302-72a62d42aebf",
			},
		},
	}
)

func loadConfig(ctx context.Context) (appConfig, error) {
	logx.Info(ctx, fmt.Sprintf("env: %s", config.Environment()))
	c, ok := defaultConfigs[config.Environment()]
	if !ok {
		return appConfig{}, errors.Wrapf(ErrNoConfigFound, "using environment %s", config.Environment())
	}

	d, err := ioutil.ReadFile("aegis.yaml")
	if err != nil {
		if !os.IsNotExist(err) {
			return c, err
		}
	} else {
		err = yaml.Unmarshal(d, &c)
		if err != nil {
			return c, err
		}
		logx.Info(ctx, "Additional config loaded from aegis.yaml")
	}

	b, err := yaml.Marshal(c)
	if err != nil {
		return appConfig{}, errors.New("Failed to marshal config")
	}

	fmt.Printf("Running with the following config:\n%s\n\n", string(b))
	return c, nil
}
