package main

import (
	"bytes"
	"context"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"path/filepath"
	"strings"
	"text/template"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/aws/session"
	graphql "github.com/graph-gophers/graphql-go"
	"github.com/pkg/errors"
	"github.com/sirupsen/logrus"

	goji "goji.io"
	"goji.io/pat"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/gometrics"
	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/safety/aegis/internal/auth"
	"code.justin.tv/safety/aegis/internal/clients"
	"code.justin.tv/safety/aegis/internal/gql"
	"code.justin.tv/safety/aegis/internal/gql/resolvers"
	"code.justin.tv/safety/aegis/internal/gql/resolvers/mutation"
	"code.justin.tv/safety/aegis/internal/gql/resolvers/query"
	"code.justin.tv/safety/aegis/internal/server"
	"code.justin.tv/safety/middleware"
	"code.justin.tv/safety/multistatter"

	twitchConfig "code.justin.tv/common/config"
)

func main() {
	err := twitchConfig.Parse()
	if err != nil {
		log.Fatalf("Failed to parse config: %v", err)
	}

	// set global statter
	err = twitchConfig.Statsd().Close()
	if err != nil {
		log.Fatalf("Error clossing connection to stats default service: %+v", err)
	}
	statter, err := multistatter.New()
	if err != nil {
		log.Fatalf("Error creating the global statter: %+v", err)
	}
	twitchConfig.SetStatsd(statter)

	// Not doing this in app_config because that introduces a cycle where we need
	// the logger before the config and the config before the logger
	if twitchConfig.Environment() != "development" {
		logx.InitDefaultLogger(logx.Config{
			RollbarToken: twitchConfig.RollbarToken(),
			RollbarEnv:   twitchConfig.Environment(),
			Formatter:    &logrus.JSONFormatter{},
		})
	} else {
		logx.InitDefaultLogger(logx.Config{
			Formatter: &logrus.TextFormatter{
				FullTimestamp:   true,
				TimestampFormat: "15:04:05.000",
			},
		})
	}

	defer logx.Wait()
	defer logx.RecoverAndLog()

	gometrics.Monitor(statter, time.Second*5)

	ctx := context.Background()
	config, err := loadConfig(ctx)
	if err != nil {
		logx.Fatal(ctx, errors.WithMessage(err, "Failed to load config file"))
	}

	// err = populateSecets(&config)
	// if err != nil {
	// logx.Fatal(ctx, errors.WithMessage(err, "Failed to populate secrets"))
	// }

	var awsSession client.ConfigProvider
	if awsRegion := twitchConfig.AwsRegion(); awsRegion != "" {
		awsSession, err = session.NewSession(aws.NewConfig().WithRegion(awsRegion))
	} else {
		awsSession, err = session.NewSession()
	}
	if err != nil {
		logx.Fatal(ctx, errors.WithMessage(err, "Failed to create AWS session"))
	}

	clients, err := clients.Create(config.Clients, awsSession)
	if err != nil {
		logx.Fatal(ctx, errors.WithMessage(err, "Failed to create clients"))
	}
	// go func() {
	// err = clients.Tracking.Start()
	// if err != nil {
	// logx.Fatal(ctx, errors.WithMessage(err, "Failed to start tracking client"))
	// }
	// }()
	// defer func() {
	// _ = clients.Tracking.Close()
	// }()

	// dbConfig := config.Datastore
	// dbConfig.Chatlog = &chatlog.Config{
	// AwsConfigProvider: awsSession,
	// }
	// dbConfig.Router.Statter = statter
	// store, err := datastore.New(dbConfig)
	// if err != nil {
	// logx.Fatal(ctx, errors.WithMessage(err, "Failed to create datastore"))
	// }

	schemaFiles, err := findAllSchemaFiles("internal/gql/schema")
	schemaDefinition := []byte{}

	for _, sf := range schemaFiles {
		var schema []byte
		if isFileTemplate(sf) {
			t, errT := template.ParseFiles(sf)
			if errT != nil {
				logx.Fatal(ctx, errors.Wrapf(err, "Failed to parse schema file %s as template", sf))
			}

			var tpl bytes.Buffer
			errT = t.Execute(&tpl, nil) // We have no template data
			if errT != nil {
				logx.Fatal(ctx, errors.Wrapf(err, "Failed to apply schema file %s as template", sf))
			}
			schema = tpl.Bytes()
		} else {
			schema, err = ioutil.ReadFile(sf)
			if err != nil {
				logx.Fatal(ctx, errors.Wrapf(err, "Failed to load schema file %s", sf))
			}
		}
		schemaDefinition = append(schemaDefinition, schema...)
	}

	queryResolver := query.Query{
		// Datastore: store,
		Clients: clients,
	}

	mutationResolver := mutation.Mutation{
		// Datastore: store,
		Clients: clients,
	}

	// Limiting to 100 parallel calls because we are not hitting the ceiling yet and
	// some of our loaders don't actually benefit past 100
	// eg channels and users
	schema, err := graphql.ParseSchema(string(schemaDefinition), &resolvers.RootResolver{
		Query:    &queryResolver,
		Mutation: &mutationResolver,
	}, graphql.MaxParallelism(100))

	if err != nil {
		logx.Fatal(ctx, errors.WithMessage(err, "Failed to parse schema"))
	}

	config.Auth.GuardianClientSecret = "" //TODO FILL THIS IN

	authProvider, err := auth.New(config.Auth)
	if err != nil {
		logx.Fatal(ctx, errors.WithMessage(err, "Failed to create auth provider"))
	}

	router := twitchserver.NewServer()
	router.Use(server.RequestIDMiddleware)
	router.Use(server.RequestLoggerMiddleware)
	router.Use(middleware.XFrameOptions(middleware.XFrameOptionsDeny))

	graphqlMux := goji.SubMux()
	// graphqlMux.Use(server.AuthorizeRequestMiddleware(store, authProvider, clients.Permissions))

	graphqlMux.Handle(pat.Post(""), &gql.Handler{Schema: schema, Statter: statter})

	// apiMux := api.NewSubMux(clients, store)
	// apiMux.Use(server.AuthorizeRequestMiddleware(store, authProvider, clients.Permissions))

	router.Handle(pat.Post("/graphql"), graphqlMux)
	router.Handle(pat.Get("/auth_exchange"), authProvider.ExchangeHandler())
	router.Handle(pat.Get("/auth_redirect"), authProvider.RedirectHandler())
	// router.Handle(pat.New("/api/*"), apiMux)
	router.HandleFunc(pat.Get("/graphql"), func(w http.ResponseWriter, r *http.Request) {
		_, requestErr := fmt.Fprint(w, server.GraphqlPlayground)
		if requestErr != nil {
			logx.Error(r.Context(), err)
		}
	})

	fi, err := os.Stat("webapp")
	if err != nil && !os.IsNotExist(err) {
		logx.Fatal(ctx, err)
	}

	if fi != nil && fi.IsDir() {
		logx.Info(ctx, "Serving static webpapp")
		var spaHandler http.Handler
		spaHandler, err = server.NewStaticFilesHandler("webapp")
		if err != nil {
			logx.Fatal(ctx, err)
		}
		router.Handle(pat.Get("/*"), spaHandler)
	}

	twitchserver.AddDefaultSignalHandlers()
	conf := twitchserver.NewConfig()
	err = twitchserver.ListenAndServe(router, conf)
	if err != nil {
		logx.Fatal(ctx, errors.WithMessage(err, "ListenAndServe failed"))
	}
}

func findAllSchemaFiles(baseDir string) ([]string, error) {
	files, err := ioutil.ReadDir(baseDir)
	if err != nil {
		return nil, errors.Wrapf(err, "Error reading %s", baseDir)
	}

	result := make([]string, 0, len(files))
	for _, file := range files {
		if file.IsDir() {
			subDirFiles, errS := findAllSchemaFiles(filepath.Join(baseDir, file.Name()))
			if errS != nil {
				return nil, errS
			}
			result = append(result, subDirFiles...)
		} else if strings.HasSuffix(file.Name(), ".graphql") || isFileTemplate(file.Name()) {
			result = append(result, filepath.Join(baseDir, file.Name()))
		}
	}

	return result, nil
}

func isFileTemplate(fileName string) bool {
	return strings.HasSuffix(fileName, ".graphql.tmpl")
}
