package clients

import (
	"context"
	"fmt"
	"math/rand"
	"time"

	"github.com/pkg/errors"

	"code.justin.tv/foundation/twitchclient"
	users "code.justin.tv/web/users-service/client/usersclient_internal"
	"code.justin.tv/web/users-service/models"
)

var (
	validLoginRunes = []rune("abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890")
)

func init() {
	rand.Seed(time.Now().UnixNano())
}

// UsersClient is thin wrapper of user service with APIs to create test data
type UsersClient struct {
	client users.InternalClient
}

// newUsersClient creates a new users client for test
func newUsersClient() (*UsersClient, error) {
	client, err := users.NewClient(twitchclient.ClientConf{
		Host: "users-service.dev.us-west2.twitch.tv",
	})
	if err != nil {
		return nil, errors.Wrap(err, "creating user client")
	}

	return &UsersClient{
		client: client,
	}, nil
}

// NewTestUser creates a new test user
func (c *UsersClient) NewTestUser(ctx context.Context) (*models.Properties, error) {
	login := generateLogin()

	return c.client.CreateUser(ctx, &models.CreateUserProperties{
		Login: login,
		Email: fmt.Sprintf("safety+%s@justin.tv", login),
		Birthday: models.Birthday{
			Day:   1,
			Month: time.January,
			Year:  1999,
		},
	}, nil)
}

// GetUserByLogin fowards a request to get a user by login to users service
func (c *UsersClient) GetUserByLogin(ctx context.Context, login string) (*models.Properties, error) {
	return c.client.GetUserByLogin(ctx, login, nil)
}

// Generates a random login
func generateLogin() string {
	b := make([]rune, 8)
	for i := range b {
		b[i] = validLoginRunes[rand.Intn(len(validLoginRunes))]
	}
	return "safety" + string(b)
}
