package api

import (
	"net/http"
	"strconv"

	"code.justin.tv/safety/datastore/models/reportreason"

	"code.justin.tv/safety/datastore/models/reportcontent"
	"code.justin.tv/safety/gateway/rpc/gateway"

	"github.com/jixwanwang/apiutils"
)

type singleReportResponse struct {
	FromUserID        string `json:"fromUserID"`
	TargetUserID      string `json:"targetUserID"`
	ReportDescription string `json:"reportDescription"`
	ReportID          string `json:"reportID"`
}

// CaseSingleReportPage creates the test artifacts for "SingleReportpage"
// This is an example for how to generate test cases for specific test case that involves
// creating new report and users
func (s *Server) CaseSingleReportPage() http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		ctx := r.Context()

		fromUser, err := s.clients.Users.GetUserByLogin(ctx, "twitchmoderation")
		if err != nil {
			serveError(r.Context(), w, http.StatusInternalServerError, "Failed to get twitch moderation user", err)
			return
		}
		fromUserID, err := strconv.ParseInt(fromUser.ID, 10, 64)
		if err != nil {
			serveError(r.Context(), w, http.StatusInternalServerError, "Failed to parse from user's id ", err)
			return
		}

		targetUser, err := s.clients.Users.NewTestUser(ctx)
		if err != nil {
			serveError(r.Context(), w, http.StatusBadRequest, "Failed to create test target user", err)
			return
		}
		targetUserID, err := strconv.ParseInt(targetUser.ID, 10, 64)
		if err != nil {
			serveError(r.Context(), w, http.StatusInternalServerError, "Failed to prase target user's id", err)
			return
		}

		description := "a quick brown fox jumps over the lazy dog"
		// Current broken due to different humanness check we put in gateway. Need to add a flag to gateway skip those checks
		// https: //jira.twitch.com/browse/SAFETY-2487
		report, err := s.clients.Gateway.Client.CreateReport(ctx, &gateway.CreateReportRequest{
			Description:  description,
			Content:      reportcontent.User,
			Reason:       reportreason.Harassment,
			FromUserId:   fromUserID,
			TargetUserId: targetUserID,
		})

		if err != nil {
			apiutils.ServeError(w, apiutils.NewErrorResponse(http.StatusInternalServerError, err.Error()))
			return
		}

		apiutils.ServeJSON(w, singleReportResponse{
			FromUserID:        fromUser.ID,
			TargetUserID:      targetUser.ID,
			ReportDescription: description,
			ReportID:          strconv.FormatInt(report.Id, 10),
		})
	})
}
