package auth

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/common/config"

	"github.com/pkg/errors"
	"golang.org/x/oauth2"

	"code.justin.tv/chat/golibs/logx"
	guardian "code.justin.tv/systems/guardian/middleware"
)

// Auth is an object that handles authentication with guardian oauth
type Auth struct {
	guardianAuth *guardian.Asiimov
	oauthConfig  *oauth2.Config
}

// New returns a new instance of Auth or an error
func New(config Config) (*Auth, error) {
	if err := config.validate(); err != nil {
		return nil, err
	}

	oauth2Config := &oauth2.Config{
		ClientID:     config.GuardianClientID,
		ClientSecret: config.GuardianClientSecret,
		Scopes:       []string{},
		Endpoint: oauth2.Endpoint{
			AuthURL:  guardian.AuthURL,
			TokenURL: guardian.TokenURL,
		},
	}

	g := guardian.New(nil, oauth2Config)
	g.NonceStore = &staticNonceStore{}

	return &Auth{
		guardianAuth: g,
		oauthConfig:  oauth2Config,
	}, nil
}

// UserInfoForAuthToken takes an AuthToken and looks up the UserInfo from guardian
func (a *Auth) UserInfoForAuthToken(ctx context.Context, accessToken string) (*UserInfo, error) {
	statter := config.Statsd()
	tc, err := a.guardianAuth.CheckToken(&oauth2.Token{
		AccessToken: accessToken,
	})
	if err != nil {
		_ = statter.Inc("auth.token.error", 1, 1)
		return nil, errors.Wrap(err, "checking token")
	}

	// Token is invalid
	if tc == nil || tc.Token == nil || tc.Token.AccessToken == "" {
		_ = statter.Inc("auth.token_invalid", 1, 1)
		logx.Info(ctx, fmt.Sprintf("Token %s is invalid", accessToken))
		return nil, nil
	}

	// Guardian doesn't allow custom token expiration at creation and
	// they suggested to check expiry upon retrieving the token
	// Set token to expire after 30 days (default 1 day + 29 days)
	if tc.Token.Expiry.Add(time.Hour * 24 * 29).Before(time.Now()) {
		_ = statter.Inc("auth.token_expired", 1, 1)
		logx.Info(ctx, fmt.Sprintf("Token %s has expired", accessToken))
		return nil, nil
	}

	// User is not found
	if tc == nil || tc.User == nil {
		_ = statter.Inc("auth.user_not_found", 1, 1)
		logx.Info(ctx, fmt.Sprintf("No user found for %s", accessToken))
		return nil, nil
	}

	return &UserInfo{
		LdapID:   tc.User.UID,
		FullName: tc.User.CN,
	}, nil
}
