package auth

import (
	"fmt"
	"net/http"

	"github.com/pkg/errors"
	"golang.org/x/oauth2"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"github.com/jixwanwang/apiutils"

	guardian "code.justin.tv/systems/guardian/middleware"
)

type redirectResponse struct {
	Location string `json:"location"`
}

// ExchangeHandler handles request to exchanges google creds for authorization token
func (a *Auth) ExchangeHandler() http.Handler {
	return a.guardianAuth.OAuth2CallbackHandler(func(err guardian.Error, w http.ResponseWriter, r *http.Request) {
		logx.Info(r.Context(), fmt.Sprintf("auth exchange err"))
		statter := config.Statsd()
		_ = statter.Inc("auth.exchange.error", 1, 1)
		_ = statter.Inc(fmt.Sprintf("auth.exchange.err_code.%v", err.StatusCode()), 1, 1)

		guardian.JSONErrorHandler(err, w, r)
	}, func(token *oauth2.Token, w http.ResponseWriter, r *http.Request) {
		logx.Info(r.Context(), fmt.Sprintf("auth exchange success"))
		_ = config.Statsd().Inc("auth.exchange.success", 1, 1)
		guardian.JSONTokenHandler(token, w, r)
	})
}

// RedirectHandler handles oauth2 redirect
func (a *Auth) RedirectHandler() http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		statter := config.Statsd()

		nonce, err := a.guardianAuth.NonceStore.Generate()
		if err != nil {
			_ = statter.Inc("auth.redirect.nounce.error", 1, 1)

			logx.Error(r.Context(), errors.WithMessage(err, "Failed to generate nonce"))
			apiutils.ServeError(w, apiutils.NewErrorResponse(http.StatusInternalServerError, "Failed to generate nonce"))
			return
		}

		_ = statter.Inc("auth.redirect.success", 1, 1)
		apiutils.ServeJSON(w, redirectResponse{Location: a.oauthConfig.AuthCodeURL(nonce, oauth2.AccessTypeOffline)})
	})
}
