// Package guardian defines Guardian client
// Guardian doesn't have a client and so we are writing our own
package guardian

import (
	"encoding/json"
	"errors"
	"io/ioutil"
	"net/http"
)

// GetLdapUsersResponse is response to GetLdapUsers call
type GetLdapUsersResponse struct {
	Data []LdapUser `json:"data"`
}

// LdapUser is an ldap user
type LdapUser struct {
	Type       string             `json:"type"`
	ID         string             `json:"id"`
	Attributes LdapUserAttributes `json:"attributes"`
}

// LdapUserAttributes is attributes for a given ldap user
type LdapUserAttributes struct {
	CN    string `json:"cn"`
	UID   string `json:"uid"`
	Email string `json:"email"`
}

// Client is a guardian client
type Client interface {
	GetLdapUsers() (*GetLdapUsersResponse, error)
}

type client struct {
	baseURL string
}

// New creates a new guardian client
func New(baseURL string) (Client, error) {
	return &client{
		baseURL: baseURL,
	}, nil
}

// GetLdapUsers fetches all ldap users
func (c *client) GetLdapUsers() (*GetLdapUsersResponse, error) {
	res, err := http.Get(c.baseURL + "/ldap/users")
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = res.Body.Close()
	}()

	users, err := ioutil.ReadAll(res.Body)
	if err != nil {
		return nil, errors.New("failed to read http response body")
	}

	var response GetLdapUsersResponse
	err = json.Unmarshal(users, &response)
	if err != nil {
		return nil, err
	}

	return &response, nil
}
