package passport

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
)

var (
	// ErrMissing2FA is returned by GetAuthyID if the requested user does not have 2fa enabled
	ErrMissing2FA = fmt.Errorf("user does not have 2FA enabled")
)

// GetAuthyIDResponse is the response of GetAuthyID response
type GetAuthyIDResponse struct {
	AuthyID   int64 `json:"authy_id"`
	CreatedAt int64 `json:"created_at"`
	UpdatedAt int64 `json:"updated_at"`
}

// GetUsersResponse is the response of GetUsers api
type GetUsersResponse struct {
	UserIDs []string `json:"user_ids"`
}

// Client is an internal passport client that provides access to authy related information
type Client interface {
	GetAuthyID(userID string) (*GetAuthyIDResponse, error)
	GetUsers(authyID int64) (*GetUsersResponse, error)
}

type client struct {
	baseURL string
}

// New creates a new Passport client. The apis in this client are not exposed in the passport
func New(baseURL string) (Client, error) {
	return &client{
		baseURL: baseURL,
	}, nil
}

// GetAuthyID fetches authy id for a given user
func (c *client) GetAuthyID(userID string) (*GetAuthyIDResponse, error) {
	res, err := http.Get(c.baseURL + fmt.Sprintf("/users/%s/2fa", userID))
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = res.Body.Close()
	}()

	users, err := ioutil.ReadAll(res.Body)
	if err != nil {
		return nil, errors.New("failed to read http response body")
	}

	var response GetAuthyIDResponse
	err = json.Unmarshal(users, &response)
	if err != nil {
		return nil, err
	}

	return &response, nil
}

// GetUsers fetches all of the users that share the same authy id
func (c *client) GetUsers(authyID int64) (*GetUsersResponse, error) {
	res, err := http.Get(c.baseURL + fmt.Sprintf("/api/v1/internal/authy_connections/%d", authyID))
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = res.Body.Close()
	}()

	users, err := ioutil.ReadAll(res.Body)
	if err != nil {
		return nil, errors.New("failed to read http response body")
	}

	var response GetUsersResponse
	err = json.Unmarshal(users, &response)
	if err != nil {
		return nil, err
	}

	return &response, nil
}
