package fileuploader

import (
	"context"
	"fmt"
	"io"

	"code.justin.tv/safety/datastore/interfaces"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3iface"
	"github.com/pkg/errors"
	uuid "github.com/satori/go.uuid"
)

// S3UploadClient uploads an image to s3
type S3UploadClient struct {
	S3Client  s3iface.S3API
	Config    Config
	Datastore interfaces.Datastore
}

// Config contains necessary paramettrs for mutations
type Config struct {
	FileUploadBucket  string
	ImageUploadPrefix string
}

// S3FileData represents data for a s3 file
type S3FileData struct {
	Name string
	URL  string
}

// UploadImage uploads an iamge to s3
func (c *S3UploadClient) UploadImage(ctx context.Context, reader io.ReadSeeker, file S3FileData, contentType string, size int64) error {
	_, err := c.S3Client.PutObjectWithContext(ctx, &s3.PutObjectInput{
		Bucket:        aws.String(c.Config.FileUploadBucket),
		Key:           aws.String(file.Name),
		Body:          reader,
		ContentLength: aws.Int64(size),
		ContentType:   aws.String(contentType),
		ACL:           aws.String(s3.ObjectCannedACLPublicRead),
	})

	return errors.Wrap(err, "saving image to s3")
}

// GenerateImageFileData returns data for the image file
func (c *S3UploadClient) GenerateImageFileData(format string) S3FileData {
	name := fmt.Sprintf("%v%v%v", c.Config.ImageUploadPrefix, uuid.NewV4().String(), format)
	url := fmt.Sprintf("https://s3-us-west-2.amazonaws.com/%s/%s", c.Config.FileUploadBucket, name)

	return S3FileData{
		Name: name,
		URL:  url,
	}
}
