package gql

import (
	"encoding/base64"
	"strconv"

	"github.com/pkg/errors"
)

// Cursor represents the scalar type from the schema.
// It is an opaque resource identifier (e.g. "10imf10nf==").
type Cursor string

// ImplementsGraphQLType informs github.com/graph-gophers/graphql-go the Cursor type
// implements the custom scalar `Cursor`.
func (*Cursor) ImplementsGraphQLType(name string) bool {
	return name == "Cursor"
}

// UnmarshalGraphQL describes how to unmarshal GraphQL input into a Cursor type.
// Required by the github.com/graph-gophers/graphql-go library.
func (c *Cursor) UnmarshalGraphQL(input interface{}) error {
	val, ok := input.(string)
	if !ok {
		return errors.New("wrong type")
	}

	*c = Cursor(val)
	return nil
}

// Offset attempts to decode an offset integer from the body of the Cursor.
func (c *Cursor) Offset() (int, error) {
	b, err := base64.URLEncoding.DecodeString(c.String())
	if err != nil {
		return 0, err
	}
	offset, err := strconv.Atoi(string(b))
	if err != nil {
		return 0, err
	}
	return offset, nil
}

// String representation of a cursor.
func (c *Cursor) String() string {
	if c == nil {
		return "<nil>"
	}
	return string(*c)
}
