package gql

import (
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"time"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/safety/aegis/internal/interfaces"
	"code.justin.tv/safety/aegis/internal/server"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/jixwanwang/apiutils"
	"github.com/pkg/errors"
	"golang.org/x/sync/errgroup"

	graphql "github.com/graph-gophers/graphql-go"
)

const (
	maxRequestSize = 1 * 1024 * 1024 // 1 mb
)

// Handler handles graphql queries, splitting up batches is needed
type Handler struct {
	Schema  interfaces.Schema
	Statter statsd.Statter
}

type queryParams struct {
	Query         string                 `json:"query"`
	OperationName string                 `json:"operationName"`
	Variables     map[string]interface{} `json:"variables"`
}

func (h *Handler) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	ctx := r.Context()

	body, err := ioutil.ReadAll(io.LimitReader(r.Body, maxRequestSize))
	if err != nil {
		_ = h.Statter.Inc("gql.request.error.parse_request", 1, 1)
		server.ServeError(ctx, w, "Failed to read request body", http.StatusBadRequest, err)
		return
	}
	if len(body) == 0 {
		_ = h.Statter.Inc("gql.request.error.zero_length", 1, 1)
		server.ServeError(ctx, w, "0 length body", http.StatusBadRequest, nil)
		return
	}

	batched := true
	queries := []*queryParams{}
	switch string(body[0]) {
	case "{":
		batched = false

		var single queryParams
		err = json.Unmarshal(body, &single)
		if err != nil {
			_ = h.Statter.Inc("gql.request.error.parse_single", 1, 1)
			server.ServeError(ctx, w, "Failed to parse signle query", http.StatusBadRequest, err)
			return
		}
		queries = append(queries, &single)
	case "[":
		err = json.Unmarshal(body, &queries)
		if err != nil {
			_ = h.Statter.Inc("gql.request.error.parse_multi", 1, 1)
			server.ServeError(ctx, w, "Failed to parse multi query", http.StatusBadRequest, err)
			return
		}
	}

	start := time.Now()
	responses := make([]*graphql.Response, len(queries))
	group, context := errgroup.WithContext(ctx)

	logx.Info(ctx, fmt.Sprintf("Got %d queries", len(queries)))
	for i, query := range queries {
		i, query := i, query // capture external variables
		group.Go(func() (err error) {
			logx.Info(ctx, fmt.Sprintf("Executing query %s with %v", query.OperationName, query.Variables))
			response := h.Schema.Exec(context, query.Query, query.OperationName, query.Variables)
			_ = h.Statter.TimingDuration(fmt.Sprintf("gql.time.%s", query.OperationName), time.Since(start), 1)

			responses[i] = response
			if response != nil {
				if response.Errors != nil {
					_ = h.Statter.Inc(fmt.Sprintf("gql.request.error.%s", query.OperationName), 1, 1)
					logx.Warn(ctx, errors.Errorf("Failed query %s with %v", query.Query, query.Variables))
					// Log the actual errors
					for _, e := range response.Errors {
						if e.ResolverError != nil {
							// do %+v to print stacktrace
							logx.Warn(ctx, fmt.Sprintf("Resolver error %+v", e.ResolverError))
						} else {
							logx.Warn(ctx, fmt.Sprintf("Query error %v", e.Error()))
						}
					}
				} else {
					_ = h.Statter.Inc(fmt.Sprintf("gql.request.success.%s", query.OperationName), 1, 1)
					logx.Info(ctx, fmt.Sprintf("Query %s succeeded", query.OperationName))
				}
			} else {
				_ = h.Statter.Inc(fmt.Sprintf("gql.request.empty.%s", query.OperationName), 1, 1)
				logx.Warn(ctx, fmt.Sprintf("Query %s has empty response", query.Query))
			}
			return nil
		})
	}
	err = group.Wait()
	if err != nil {
		_ = h.Statter.Inc("gql.request.errors", 1, 1)
		server.ServeError(ctx, w, "Failed to execute queries", http.StatusInternalServerError, err)
		return
	}

	if batched {
		apiutils.ServeJSON(w, responses)
	} else {
		apiutils.ServeJSON(w, responses[0])
	}
}
