package gql

import (
	"context"
	"fmt"
	"net/http"
	"net/http/httptest"
	"strings"
	"sync/atomic"
	"testing"

	"github.com/cactus/go-statsd-client/statsd"
	graphql "github.com/graph-gophers/graphql-go"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"code.justin.tv/safety/aegis/internal/mocks"
)

const dummyQuery = `{ dummy { dummy { id } } }`

func TestNoRequest(t *testing.T) {
	schema := &mocks.FakeSchema{}
	statter, err := statsd.NewNoopClient()
	require.NoError(t, err)
	handler := &Handler{Schema: schema, Statter: statter}

	server := httptest.NewServer(handler)
	defer server.Close()

	resp, _ := http.Get(server.URL)
	assert.Equal(t, http.StatusBadRequest, resp.StatusCode)
}

func TestSingleRequest(t *testing.T) {
	schema := &mocks.FakeSchema{}
	statter, err := statsd.NewNoopClient()
	require.NoError(t, err)
	handler := &Handler{Schema: schema, Statter: statter}

	server := httptest.NewServer(handler)
	defer server.Close()

	query := fmt.Sprintf(`{"query": "%s" }`, dummyQuery)
	resp, err := http.Post(server.URL, "application/json", strings.NewReader(query))
	assert.NoError(t, err)
	require.NotNil(t, resp)
	assert.Equal(t, http.StatusOK, resp.StatusCode)

	assert.Len(t, schema.Invocations(), 1)
}

func TestMultiRequest(t *testing.T) {
	schema := &mocks.FakeSchema{}
	statter, err := statsd.NewNoopClient()
	require.NoError(t, err)
	handler := &Handler{Schema: schema, Statter: statter}

	var invocations int32
	schema.ExecStub = func(ctx context.Context, queryString string, operationName string, variables map[string]interface{}) *graphql.Response {
		atomic.AddInt32(&invocations, 1)
		assert.Equal(t, queryString, dummyQuery)
		return &graphql.Response{}
	}

	server := httptest.NewServer(handler)
	defer server.Close()

	query := fmt.Sprintf(`[{"query": "%s" }, {"query": "%s" }]`, dummyQuery, dummyQuery)
	resp, err := http.Post(server.URL, "application/json", strings.NewReader(query))
	assert.NoError(t, err)
	require.NotNil(t, resp)
	assert.Equal(t, http.StatusOK, resp.StatusCode)

	assert.EqualValues(t, invocations, 2)
}
