package images

import (
	"fmt"

	"github.com/pkg/errors"

	// Register following image types
	_ "image/jpeg"
	_ "image/png"
)

var (
	// ErrContentTypeNotAllowed  is returned when the content type is not allowed
	ErrContentTypeNotAllowed = fmt.Errorf("content type is not allowed")
)

var (
	// contentTypeToExtension is a map of content type to extensions
	contentTypeToExtension = map[string]string{
		"image/jpeg": ".jpg",
		"image/png":  ".png",
	}
)

// ExtensionByType returns the first matched extension for the content type,
// or an error if none matched
func ExtensionByType(contentType string) (string, error) {
	ext, ok := contentTypeToExtension[contentType]
	if !ok {
		return "", errors.Wrapf(ErrContentTypeNotAllowed, "content type is %v", contentType)
	}

	return ext, nil
}
