package permissions

import (
	"context"

	beefcake "code.justin.tv/amzn/TwitchBeefcakeTwirp"
)

// Permissions is a factory for UserPermissions objects that are created
// by Aegis per request
type Permissions struct {
	beefcakeClient beefcake.TwitchBeefcake
}

// UserPermissions is beefcake permissions
type UserPermissions interface {
	IsPermitted(id string) bool
}

type userPermissions struct {
	perms map[string]bool
}

// New returns a new instance of Permissions or an error
func New(beefcakeClient beefcake.TwitchBeefcake) (*Permissions, error) {
	return &Permissions{
		beefcakeClient: beefcakeClient,
	}, nil
}

// WithPermissionsObject returns a new context with permissions object
func (p *Permissions) WithPermissionsObject(ctx context.Context, ldap string) (context.Context, error) {
	response, err := p.beefcakeClient.GetUser(ctx, &beefcake.GetUserRequest{Id: ldap})
	if err != nil {
		return nil, err
	}

	perms := make(map[string]bool, len(response.Permissions))
	for _, i := range response.Permissions {
		perms[i.Id] = true
	}
	beefcakePermissions := userPermissions{
		perms: perms,
	}

	return context.WithValue(ctx, MyPermissions, beefcakePermissions), nil
}

// UsersWithReportAccess returns a list of ldap IDs with access to read reports
func (p *Permissions) UsersWithReportAccess(ctx context.Context) ([]string, error) {
	//generate client needs a second paramatter, but it does nothing, so we pass a nil
	response, err := p.beefcakeClient.GetLegacyPermission(ctx, &beefcake.GetLegacyPermissionRequest{Id: legacyReportReadPermission})
	if err != nil {
		return nil, err
	}

	permissionUsers := response.GetUsers()
	ldapUsersUnique := make(map[string]interface{})
	ldapUsers := make([]string, 0, len(permissionUsers))
	for _, user := range permissionUsers {
		if _, ok := ldapUsersUnique[user.Id]; ok {
			continue
		}
		ldapUsers = append(ldapUsers, user.Id)
		ldapUsersUnique[user.Id] = nil
	}

	return ldapUsers, nil
}

// IsPermitted returns true if the user has permissions for the resource
// specified in `id`.
func (u userPermissions) IsPermitted(id string) bool {
	actions, ok := u.perms[id]
	if !ok {
		return false
	}
	return actions
}
