package permissions

import (
	"context"

	"github.com/pkg/errors"

	"code.justin.tv/safety/datastore/models"
)

// CanReadReport returns if a user can read a specific report
//
// BeefcakePermission: ReportRead
//
// Params:
//	adminID - The ID of the user from the users table. Used to allow users to read reports assigned
//            to them regardless of beefcake permissions
func CanReadReport(ctx context.Context, adminID int64, report *models.Report) (bool, error) {
	p, ok := ctx.Value(MyPermissions).(UserPermissions)
	if !ok {
		return false, errors.WithStack(ErrPermissionsMissing)
	}

	// Always allow if user has the permission to read report
	if p.IsPermitted(ReportRead) {
		return true, nil
	}

	return report != nil && report.AssignedTo != nil && *report.AssignedTo == adminID, nil
}

// CanWriteReport returns if a user can write to a specific report
//
// BeefcakePermission: ReportWrite
//
// Params:
//	adminID - The ID of the user from the users table. Used to allow users to write reports assigned
//            to them regardless of beefcake permissions
func CanWriteReport(ctx context.Context, adminID int64, report *models.Report) (bool, error) {
	p, ok := ctx.Value(MyPermissions).(UserPermissions)
	if !ok {
		return false, errors.WithStack(ErrPermissionsMissing)
	}

	// Always allow if user has the permission to write report
	if p.IsPermitted(ReportWrite) {
		return true, nil
	}

	return report != nil && report.AssignedTo != nil && *report.AssignedTo == adminID, nil
}
