package permissions

import (
	"context"
	"testing"

	"code.justin.tv/safety/datastore/models"

	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
)

func AllowFor(perm Permission) UserPermissions {
	perms := userPermissionFake{value: true}
	return perms
}

func AllowNone() UserPermissions {
	perms := userPermissionFake{value: false}

	return perms
}

func TestUserCanReadReport(t *testing.T) {
	adminID := int64(123)

	testCases := []struct {
		permissions UserPermissions
		report      *models.Report
		canRead     bool
	}{
		{
			permissions: AllowNone(),
			report: &models.Report{
				AssignedTo: &adminID,
			},
			canRead: true,
		},
		{
			permissions: AllowFor(ReportRead),
			report: &models.Report{
				AssignedTo: nil,
			},
			canRead: true,
		},
		{
			permissions: AllowNone(),
			report: &models.Report{
				AssignedTo: nil,
			},
			canRead: false,
		},
		{
			permissions: AllowNone(),
			report:      nil,
			canRead:     false,
		},
	}

	for _, testCase := range testCases {
		ctx := context.WithValue(context.Background(), MyPermissions, testCase.permissions)

		perm, err := CanReadReport(ctx, adminID, testCase.report)
		assert.NoError(t, err)
		assert.Equal(t, testCase.canRead, perm)
	}
}

func TestUserCanReadReportErrors(t *testing.T) {
	ctx := context.Background()
	perm, err := CanReadReport(ctx, 123, nil)
	assert.Equal(t, ErrPermissionsMissing, errors.Cause(err))
	assert.False(t, perm)
}

func TestUserCanWriteReport(t *testing.T) {
	adminID := int64(123)

	testCases := []struct {
		permissions UserPermissions
		report      *models.Report
		canRead     bool
	}{
		{
			permissions: AllowNone(),
			report: &models.Report{
				AssignedTo: &adminID,
			},
			canRead: true,
		},
		{
			permissions: AllowFor(ReportWrite),
			report: &models.Report{
				AssignedTo: nil,
			},
			canRead: true,
		},
		{
			permissions: AllowNone(),
			report: &models.Report{
				AssignedTo: nil,
			},
			canRead: false,
		},
		{
			permissions: AllowNone(),
			report:      nil,
			canRead:     false,
		},
	}

	for _, testCase := range testCases {
		ctx := context.WithValue(context.Background(), MyPermissions, testCase.permissions)

		perm, err := CanWriteReport(ctx, adminID, testCase.report)
		assert.NoError(t, err)
		assert.Equal(t, testCase.canRead, perm)
	}
}

func TestUserCanWriteReportErrors(t *testing.T) {
	ctx := context.Background()
	perm, err := CanWriteReport(ctx, 123, nil)
	assert.Equal(t, ErrPermissionsMissing, errors.Cause(err))
	assert.False(t, perm)
}
