package reports

import (
	"code.justin.tv/safety/datastore/models"
)

// GroupByTargetUser groups reports by target user while preserving the order of target user id
// limit of X will limits up to X users or X reports shown, which ever comes first
// eg.
//   (r1,t1), (r2,t2), (r3,t1), (r4,t3), (r5, t1) with limit 2 will output (r1,t1), (r3,t1), (r5,t1) // all 3 belongs to 1 user
//   (r1,t1), (r2,t2), (r3,t1), (r4,t3), (r5, t1) with limit 3 will output (r1,t1), (r3,t1), (r5,t1)
//   (r1,t1), (r2,t2), (r3,t1), (r4,t3), (r5, t1) with limit 4 will output (r1,t1), (r3,t1), (r5,t1), (r2,t2)
func GroupByTargetUser(reports []*models.Report, limit int) []*models.Report {
	targetUserOrder := make([]int, 0, limit) // map does not preserve order
	reportGroups := map[int][]*models.Report{}

	for _, report := range reports {
		group, ok := reportGroups[*report.TargetUserID]
		if !ok {
			if len(targetUserOrder) == limit { // We've hit maximum number of users; don't add anymore users
				continue
			}
			group = []*models.Report{}
			targetUserOrder = append(targetUserOrder, *report.TargetUserID)
		}
		group = append(group, report)
		reportGroups[*report.TargetUserID] = group
	}

	groupedReports := make([]*models.Report, 0, len(reports))
	for _, id := range targetUserOrder {
		groupedReports = append(groupedReports, reportGroups[id]...)
		if len(groupedReports) >= limit { // We've hit maxmimum number of report; don't add next users
			break
		}
	}

	return groupedReports
}
