package reports

import (
	"math"
	"time"

	"code.justin.tv/safety/datastore/models"

	"code.justin.tv/safety/datastore/models/reportcontent"
	"code.justin.tv/safety/datastore/models/reportreason"
	"code.justin.tv/safety/datastore/models/reportusertype"
)

const (
	twitchModerationUserID = 142101518 // The user we use for automated reports
	sintorraUserID         = 29549896  // The one who spams admins with a lot of user name reports (useful but not important)
)

// Priority returns the priority of the report. Higher number means more important
func Priority(r *models.Report) float64 {
	return contentPriority(r) * reasonPriority(r) * agePriority(r) * fromUserTypePriority(r) * fromUserPriority(r) * gamePriority(r)
}

func contentPriority(report *models.Report) float64 {
	switch *report.Content {
	case reportcontent.Whisper, reportcontent.CurseWhisper:
		return 0.5
	default:
		return 1.0
	}
}
func reasonPriority(report *models.Report) float64 {
	switch *report.Reason {
	case reportreason.SexualViolence, reportreason.SexualContent, reportreason.SelfHarm:
		return 5.0
	case reportreason.Gore, reportreason.Prohibited:
		return 4.0
	case reportreason.BitsViolation, reportreason.Underaged:
		return 2.0
	case reportreason.Spam, reportreason.Cheating, reportreason.OffensiveUsername, reportreason.Miscategorized:
		return 1.0
	default:
		return 3.0
	}
}

func agePriority(report *models.Report) float64 {
	return 1.0 + math.Floor(float64(time.Since(report.CreatedAt))/float64(time.Minute)/30.0)
}

func fromUserTypePriority(report *models.Report) float64 {
	if report.FromUserType == nil {
		return 1.0
	}

	switch *report.FromUserType {
	case reportusertype.Ceo, reportusertype.Exec:
		return 100.0
	case reportusertype.Staff, reportusertype.Admin, reportusertype.GlobalMod:
		return 3.0
	default:
		return 1.0
	}
}

func fromUserPriority(report *models.Report) float64 {
	switch *report.FromUserID {
	case twitchModerationUserID:
		return 100.0
	case sintorraUserID:
		return 0.5
	default:
		return 1.0
	}
}

func gamePriority(report *models.Report) float64 {
	if report.TargetUserGame == nil {
		return 1.0
	}

	switch *report.TargetUserGame {
	case "Artifact":
		return 50.0
	default:
		return 1.0
	}
}
