package reports

import (
	"testing"
	"time"

	"code.justin.tv/safety/datastore/models"
	"code.justin.tv/safety/datastore/models/reportcontent"
	"code.justin.tv/safety/datastore/models/reportreason"
	"code.justin.tv/safety/datastore/models/reportusertype"

	"github.com/stretchr/testify/assert"
)

func TestFromUserTypePriority(t *testing.T) {
	testCases := []struct {
		fromUserTypes []string
		priority      float64
	}{
		{
			fromUserTypes: []string{reportusertype.Ceo, reportusertype.Exec},
			priority:      100.0,
		}, {
			fromUserTypes: []string{reportusertype.Staff, reportusertype.Admin, reportusertype.GlobalMod},
			priority:      3.0,
		}, {
			fromUserTypes: []string{"random", "pleb"},
			priority:      1.0,
		},
	}

	for _, testCase := range testCases {
		for _, userType := range testCase.fromUserTypes {
			assert.Equal(t, testCase.priority, fromUserTypePriority(&models.Report{
				FromUserType: &userType,
			}))
		}
	}
}

func TestContentPriority(t *testing.T) {
	testCases := []struct {
		contents []string
		priority float64
	}{
		{
			contents: []string{reportcontent.Whisper, reportcontent.CurseWhisper},
			priority: 0.5,
		}, {
			contents: []string{"random", "content"},
			priority: 1.0,
		},
	}

	for _, testCase := range testCases {
		for _, content := range testCase.contents {
			assert.Equal(t, testCase.priority, contentPriority(&models.Report{
				Content: &content,
			}))
		}
	}
}

func TestFromUserTypePriorityNilType(t *testing.T) {
	assert.Equal(t, 1.0, fromUserTypePriority(&models.Report{
		FromUserType: nil,
	}))
}
func TestReasonPriority(t *testing.T) {
	testCases := []struct {
		reasons  []string
		priority float64
	}{
		{
			reasons:  []string{reportreason.SexualViolence, reportreason.SexualContent, reportreason.SelfHarm},
			priority: 5.0,
		}, {
			reasons:  []string{reportreason.Gore, reportreason.Prohibited},
			priority: 4.0,
		}, {
			reasons:  []string{reportreason.BitsViolation, reportreason.Underaged},
			priority: 2.0,
		}, {
			reasons:  []string{reportreason.Spam, reportreason.Cheating, reportreason.OffensiveUsername, reportreason.Miscategorized},
			priority: 1.0,
		}, {
			reasons:  []string{"random", "reason"},
			priority: 3.0,
		},
	}

	for _, testCase := range testCases {
		for _, reason := range testCase.reasons {
			assert.Equal(t, testCase.priority, reasonPriority(&models.Report{
				Reason: &reason,
			}))
		}
	}
}
func TestFromUserPriority(t *testing.T) {
	testCases := []struct {
		FromUserID int
		priority   float64
	}{
		{
			FromUserID: twitchModerationUserID,
			priority:   100.0,
		},
		{
			FromUserID: sintorraUserID,
			priority:   0.5,
		},
		{
			FromUserID: 1234567,
			priority:   1.0,
		},
	}
	for _, testCase := range testCases {
		assert.Equal(t, testCase.priority, fromUserPriority(&models.Report{
			FromUserID: &testCase.FromUserID,
		}))
	}
}

func TestAgePriority(t *testing.T) {
	testCases := []struct {
		createdAt time.Time
		priority  float64
	}{
		{
			createdAt: time.Now(),
			priority:  1.0,
		},
		{
			createdAt: time.Now().Add(-1*time.Hour + time.Second),
			priority:  2.0,
		},
		{
			createdAt: time.Now().Add(-10*time.Hour + time.Second),
			priority:  20.0,
		},
	}
	for _, testCase := range testCases {
		assert.Equal(t, testCase.priority, agePriority(&models.Report{
			CreatedAt: testCase.createdAt,
		}))
	}
}

func TestGamePriority(t *testing.T) {
	testCases := []struct {
		game     string
		priority float64
	}{
		{
			game:     "Artifact",
			priority: 50.0,
		},
		{
			game:     "ArtificialFact",
			priority: 1.0,
		},
	}
	for _, testCase := range testCases {
		assert.Equal(t, testCase.priority, gamePriority(&models.Report{
			TargetUserGame: &testCase.game,
		}))
	}

	assert.Equal(t, 1.0, gamePriority(&models.Report{}))
}
