package server

import (
	"context"
	"fmt"
	"math/rand"
	"net/http"
	"strings"
	"time"

	"github.com/pkg/errors"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/safety/aegis/internal/auth"
	"code.justin.tv/safety/aegis/internal/ctxkeys"
	"code.justin.tv/safety/aegis/internal/logkeys"
	"code.justin.tv/safety/aegis/internal/permissions"
	datastore "code.justin.tv/safety/datastore/interfaces"
)

const (
	bearer          = "Bearer "
	requestIDHeader = "x-request-id"
)

func init() {
	rand.Seed(time.Now().UnixNano())
}

// RequestIDMiddleware injects the request id into context and response header
func RequestIDMiddleware(inner http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		requestID := fmt.Sprintf("%08x", rand.Uint32())
		r = r.WithContext(context.WithValue(r.Context(), ctxkeys.RequestID, requestID))
		w.Header()[requestIDHeader] = []string{requestID}
		inner.ServeHTTP(w, r)
	})
}

// RequestLoggerMiddleware logs the request
func RequestLoggerMiddleware(inner http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		logx.Info(r.Context(), fmt.Sprintf("%s -> %s %q", r.RemoteAddr, r.Method, r.URL.String()))
		inner.ServeHTTP(w, r)
	})
}

// AuthorizeRequestMiddleware creates a middleware that authorizes request and put user information in context
func AuthorizeRequestMiddleware(ds datastore.Datastore, authProvider *auth.Auth, permissions *permissions.Permissions) func(http.Handler) http.Handler {
	return func(inner http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			ctx := r.Context()
			statter := config.Statsd()
			auth := r.Header["Authorization"]
			if len(auth) == 0 {
				_ = statter.Inc("request.auth.error.header.missing", 1, 1)
				logx.Warn(ctx, errors.Errorf("Auth header is missing"))
				ServeError(ctx, w, "Auth header is missing", http.StatusForbidden, nil)
				return
			}
			if !strings.HasPrefix(auth[0], bearer) {
				_ = statter.Inc("request.auth.error.header.invalid_format", 1, 1)
				logx.Warn(ctx, errors.Errorf("Token is missing from auth header"))
				ServeError(ctx, w, "Token is missing from auth header", http.StatusForbidden, nil)
				return
			}
			accessToken := strings.TrimPrefix(auth[0], bearer)
			ui, err := authProvider.UserInfoForAuthToken(ctx, accessToken)
			if err != nil {
				_ = statter.Inc("request.auth.error.user_info.error", 1, 1)
				logx.Warn(ctx, errors.WithMessage(err, "unable to retrieve token"))
				ServeError(ctx, w, "Failed to retrieve user info from auth provider", http.StatusInternalServerError, err)
				return
			}
			if ui == nil {
				_ = statter.Inc("request.auth.error.user_info.missing", 1, 1)
				logx.Warn(ctx, errors.Errorf("Auth provider returned no user for %s", accessToken))
				ServeError(ctx, w, "Invalid auth token", http.StatusForbidden, err)
				return
			}

			tx, err := ds.Begin()
			if err != nil {
				_ = statter.Inc("request.auth.error.datastore", 1, 1)
				ServeError(ctx, w, "Failed to begin transaction", http.StatusInternalServerError, err)
				return
			}
			defer tx.SafeRollback(ctx)

			user, err := tx.AdminByLdap(ctx, ui.LdapID)
			if err != nil {
				_ = statter.Inc("request.auth.error.datastore.adminByLdap", 1, 1)
				logx.Warn(ctx, errors.WithMessage(err, "unable to retrieve ldap from db"))
				ServeError(ctx, w, "Failed to get admin by ldap", http.StatusInternalServerError, err)
				return
			}

			if user == nil {
				_ = statter.Inc("request.auth.user_not_found", 1, 1)
				// TODO: Create user
				logx.Warn(ctx, errors.Errorf("User for %s does not exist in our db", ui.LdapID))
				ServeError(ctx, w, fmt.Sprintf("User %s is not in system", ui.LdapID), http.StatusForbidden, err)
				return
			}

			if !user.Enabled {
				_ = statter.Inc("request.auth.user_disabled", 1, 1)
				logx.Warn(ctx, errors.Errorf("User %s is not enabled", ui.LdapID))
				ServeError(ctx, w, fmt.Sprintf("User %s is not enabled", ui.LdapID), http.StatusForbidden, err)
				return
			}

			ctx, err = permissions.WithPermissionsObject(ctx, user.LdapID)
			if err != nil {
				_ = statter.Inc("request.auth.error.retrieve_permission", 1, 1)
				logx.Error(ctx, "Error retrieving user permissions", logx.Fields{
					"error": err,
				})
				ServeError(ctx, w, "Error retrieving user permissions", http.StatusInternalServerError, err)
				return
			}

			requestID := ctx.Value(ctxkeys.RequestID)
			ctx = context.WithValue(ctx, ctxkeys.MyUser, user)
			ctx = context.WithValue(ctx, ctxkeys.MyUserInfo, ui)
			ctx = logx.WithFields(ctx, logx.Fields{
				logkeys.AdminLdap: user.LdapID,
				logkeys.AdminID:   user.ID,
				logkeys.RequestID: requestID,
			})

			r = r.WithContext(ctx)

			logx.Info(ctx, fmt.Sprintf("Request %s by %d - %s", requestID, user.ID, user.LdapID))

			_ = config.Statsd().Inc("request.auth.success", 1, 0.1)
			inner.ServeHTTP(w, r)
		})
	}
}
