package server

import (
	"bytes"
	"compress/gzip"
	"context"
	"crypto/sha256"
	"encoding/hex"
	"fmt"
	"io"
	"io/ioutil"
	"mime"
	"net/http"
	"os"
	"path/filepath"
	"strings"

	"code.justin.tv/chat/golibs/logx"

	"github.com/golang/gddo/httputil/header"
	"github.com/pkg/errors"
)

const (
	vary            = "Vary"
	etag            = "Etag"
	acceptEncoding  = "Accept-Encoding"
	contentEncoding = "Content-Encoding"
	contentType     = "Content-Type"
)

type file struct {
	info        os.FileInfo
	contents    []byte
	gzContent   []byte
	eTag        string
	contentType string
}

// staticFilesHandler is a handler designed to server static files to a single page webapp.
//
// If a request for a file exists, the contents of that file are served, if the file does not
// exist, the contents of index.html are returned allowing the SPA to handle the 404.
//
// Files are cached in memory, including gzipped versions, and served up based on the Accept-Encoding
// http header.
//
// TODO: Consider watching for filesystem events - see https://github.com/fsnotify/fsnotify/blob/master/example_test.go
type staticFilesHandler struct {
	files map[string]file
}

func clientAcceptsGzip(h http.Header) bool {
	specs := header.ParseAccept(h, acceptEncoding)
	for _, s := range specs {
		if s.Value == "gzip" && s.Q > 0.0 {
			return true
		}
	}
	return false
}

func (sr *staticFilesHandler) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	found, ok := sr.files[r.URL.Path]
	if !ok {
		found, ok = sr.files["/index.html"]
	}
	if !ok {
		http.Error(w, "404 page not found", http.StatusNotFound)
		return
	}
	contents := found.contents
	if clientAcceptsGzip(r.Header) {
		contents = found.gzContent
		w.Header().Set(contentEncoding, "gzip")
	}
	w.Header().Set(vary, acceptEncoding)
	w.Header().Set(contentType, found.contentType)
	w.Header().Set(etag, found.eTag)

	http.ServeContent(w, r, found.info.Name(), found.info.ModTime(), bytes.NewReader(contents))
}

// NewStaticFilesHandler creates a new file handler that handles static files
func NewStaticFilesHandler(root string) (http.Handler, error) {
	files := map[string]file{}

	err := filepath.Walk(root, func(path string, info os.FileInfo, err error) error {
		if !info.IsDir() {
			contents, innerErr := ioutil.ReadFile(path)
			if innerErr != nil {
				return errors.Wrapf(innerErr, "reading %s", path)
			}

			var buf bytes.Buffer
			gzw := gzip.NewWriter(&buf)
			_, err := io.Copy(gzw, bytes.NewReader(contents))
			if err != nil {
				return errors.Wrap(err, "calling io.Copy to compress contents")
			}
			err = gzw.Close()
			if err != nil {
				return errors.Wrap(err, "closing gzip writer")
			}

			hasher := sha256.New()
			_, err = hasher.Write(buf.Bytes())
			if err != nil {
				return errors.Wrap(err, "writing contents to get sha256")
			}

			contentType := mime.TypeByExtension(filepath.Ext(path))
			if contentType == "" {
				contentType = http.DetectContentType(contents)
			}

			name := strings.TrimPrefix(path, root)
			logx.Info(context.Background(), "serving file", logx.Fields{
				"contentType":  contentType,
				"localPath":    path,
				"relativeName": name,
				"eTag":         hex.EncodeToString(hasher.Sum(nil)),
			})

			files[name] = file{
				info:        info,
				contents:    contents,
				gzContent:   buf.Bytes(),
				eTag:        fmt.Sprintf("%q", hex.EncodeToString(hasher.Sum(nil))),
				contentType: contentType,
			}
		}
		return nil
	})
	if err != nil {
		return nil, errors.Wrapf(err, "walking %s", root)
	}
	return &staticFilesHandler{files}, nil
}
