package tracking

import (
	"code.justin.tv/hygienic/spade"

	"code.justin.tv/safety/datastore/models"
)

const (
	// aegisAutoresolveRuleCreate is the spade event name for creating new auto resolve rules
	aegisAutoresolveRuleCreate = "aegis_autoresolve_rule_create"

	// Autoresolve user types
	autoresolveUserIs  = "is"
	autoresolveUserNot = "not"
)

// AutoresolveRuleCreate is an event for creating autoresolve rules
type AutoresolveRuleCreate struct {
	AdminID            int64   `json:"admin_id"`
	AutoresolveRuleID  int64   `json:"autoresolve_rule_id"`
	CommentID          int64   `json:"comment_id"`
	Environment        string  `json:"environment"`
	ReportID           int64   `json:"report_id"`
	SetUpSecond        int64   `json:"set_up_second"`
	FromUserID         *int64  `json:"from_user_id,omitempty"`
	FromUserRuleType   *string `json:"from_user_rule_type,omitempty"`
	TargetUserID       *int64  `json:"target_user_id,omitempty"`
	TargetUserRuleType *string `json:"target_user_rule_type,omitempty"`
	RestrictToContent  *string `json:"restrict_to_content,omitempty"`
	RestrictToReason   *string `json:"restrict_to_reason,omitempty"`
}

// TrackAutoresolveRuleCreate tracks when an admin creates a new auto resolve rule
func (c *Client) TrackAutoresolveRuleCreate(autoresolve models.AutoResolve) {
	var targetUserID, fromUserID *int64
	var targetUserType, fromUserType *string

	if autoresolve.TargetUserID != nil {
		targetUserID, targetUserType = autoresolveUserType(int64(*autoresolve.TargetUserID))
	}

	if autoresolve.FromUserID != nil {
		fromUserID, fromUserType = autoresolveUserType(int64(*autoresolve.FromUserID))
	}

	c.spade.QueueEvents(spade.Event{
		Name: aegisAutoresolveRuleCreate,
		Properties: AutoresolveRuleCreate{
			AdminID:            *autoresolve.CreatedBy,
			AutoresolveRuleID:  autoresolve.ID,
			Environment:        c.environment,
			SetUpSecond:        int64(autoresolve.ResolveUtil.Sub(autoresolve.CreatedAt).Seconds()),
			TargetUserID:       targetUserID,
			TargetUserRuleType: targetUserType,
			FromUserID:         fromUserID,
			FromUserRuleType:   fromUserType,
			RestrictToContent:  autoresolve.Content,
			RestrictToReason:   autoresolve.Reason,
		},
	})
}

// TODO https://jira.twitch.com/browse/SAFETY-1523
func autoresolveUserType(id int64) (*int64, *string) {
	var t string
	var normalizedID int64

	if id > 0 {
		t = autoresolveUserIs
		normalizedID = id
	} else {
		t = autoresolveUserNot
		normalizedID = id * -1
	}
	return &normalizedID, &t
}
