package tracking

import (
	"code.justin.tv/hygienic/spade"

	"github.com/aws/aws-sdk-go/aws"

	"code.justin.tv/safety/datastore/models"
	"code.justin.tv/safety/datastore/models/report/queue"
)

const (
	// leviathanReportAudit is the spade event name for leviathan report audits
	leviathanReportAudit = "leviathan_report_audit"
)

// ReportAudit is a event for a report audit
type ReportAudit struct {
	Action           *string `json:"action"`
	ActionBy         *int64  `json:"action_by"`
	AssignedTo       *int64  `json:"assigned_to,omitempty"`
	AutoResolveID    *int64  `json:"autoresolve_id,omitempty"`
	DestinationQueue *string `json:"destination_queue,omitempty"`
	Environment      string  `json:"environment"`
	FromQueue        *string `json:"from_queue,omitempty"`
	HoldID           *int64  `json:"hold_id,omitempty"`
	ReportID         *int64  `json:"report_id"`
}

// TrackReportAudit tracks a report audit event.
// Use TrackReportAuditReassignAdmin for admin reassignment and TrackReportAuditReassignQueue for queue reassignment
func (c *Client) TrackReportAudit(reportAudit models.ReportAudit) {
	c.spade.QueueEvents(spade.Event{
		Name: leviathanReportAudit,
		Properties: ReportAudit{
			ReportID:      reportAudit.ReportID,
			Action:        reportAudit.Action,
			ActionBy:      reportAudit.ActionBy,
			AutoResolveID: reportAudit.AutoResolveID,
			Environment:   c.environment,
			HoldID:        reportAudit.HoldID,
		},
	})
}

// TrackReportAuditReassignAdmin tracks a report audit for reassigning admin event
func (c *Client) TrackReportAuditReassignAdmin(reportAudit models.ReportAudit, reassignedToID *int64) {
	c.spade.QueueEvents(spade.Event{
		Name: leviathanReportAudit,
		Properties: ReportAudit{
			Action:      reportAudit.Action,
			ActionBy:    reportAudit.ActionBy,
			AssignedTo:  reassignedToID,
			Environment: c.environment,
			ReportID:    reportAudit.ReportID,
		},
	})
}

// TrackReportAuditReassignQueue tracks a report audit for queue event
func (c *Client) TrackReportAuditReassignQueue(reportAudit models.ReportAudit, oldQueue int64, newQueue int64) {
	c.spade.QueueEvents(spade.Event{
		Name: leviathanReportAudit,
		Properties: ReportAudit{
			Action:           reportAudit.Action,
			ActionBy:         reportAudit.ActionBy,
			Environment:      c.environment,
			FromQueue:        aws.String(queue.ValueToNameMap[oldQueue]),
			DestinationQueue: aws.String(queue.ValueToNameMap[newQueue]),
			ReportID:         reportAudit.ReportID,
		},
	})
}
