package tracking

import (
	"testing"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	spade "code.justin.tv/safety/aegis/internal/tracking/mocks"
	"code.justin.tv/safety/datastore/models"
	"code.justin.tv/safety/datastore/models/report/queue"
)

const (
	testEnvironment = "unit_test"
)

func TestTrackReportAudit(t *testing.T) {
	c, s := testClient()

	audit := models.ReportAudit{
		Action:        aws.String("action"),
		ActionBy:      aws.Int64(123),
		AutoResolveID: aws.Int64(234),
		HoldID:        aws.Int64(345),
		ReportID:      aws.Int64(456),
	}

	c.TrackReportAudit(audit)

	events := s.GetEvents()
	require.Equal(t, 1, len(events))
	event := events[0]
	assert.Equal(t, leviathanReportAudit, event.Name)
	assert.Equal(t, ReportAudit{
		Action:        audit.Action,
		ActionBy:      audit.ActionBy,
		AutoResolveID: audit.AutoResolveID,
		Environment:   testEnvironment,
		HoldID:        audit.HoldID,
		ReportID:      audit.ReportID,
	}, event.Properties)
}

func TestTrackReportAuditForAdminAssignment(t *testing.T) {
	c, s := testClient()

	adminID := int64(345)
	audit := models.ReportAudit{
		Action:   aws.String("action"),
		ActionBy: aws.Int64(123),
		ReportID: aws.Int64(234),
	}

	c.TrackReportAuditReassignAdmin(audit, &adminID)

	events := s.GetEvents()
	require.Equal(t, 1, len(events))
	event := events[0]
	assert.Equal(t, leviathanReportAudit, event.Name)
	assert.Equal(t, ReportAudit{
		Action:      audit.Action,
		ActionBy:    audit.ActionBy,
		AssignedTo:  &adminID,
		Environment: testEnvironment,
		ReportID:    audit.ReportID,
	}, event.Properties)
}

func TestTrackReportAuditReassignQueue(t *testing.T) {
	c, s := testClient()

	audit := models.ReportAudit{
		Action:   aws.String("action"),
		ActionBy: aws.Int64(123),
		ReportID: aws.Int64(234),
	}

	c.TrackReportAuditReassignQueue(audit, queue.Main, queue.Investigative)

	events := s.GetEvents()
	require.Equal(t, 1, len(events))
	event := events[0]
	assert.Equal(t, leviathanReportAudit, event.Name)
	assert.EqualValues(t, ReportAudit{
		Action:           audit.Action,
		ActionBy:         audit.ActionBy,
		DestinationQueue: aws.String(queue.ValueToNameMap[queue.Investigative]),
		Environment:      testEnvironment,
		FromQueue:        aws.String(queue.ValueToNameMap[queue.Main]),
		ReportID:         audit.ReportID,
	}, event.Properties)
}

// testClient returns an async client used for testing with
// mocked out spade client that can be used to verify the tracking event
func testClient() (*Client, *spade.FakeSpade) {
	client := NewFake(testEnvironment)

	return client, client.spade.(*spade.FakeSpade)
}
