package tracking

import (
	"code.justin.tv/hygienic/spade"

	"code.justin.tv/safety/datastore/models"
)

const (
	// aegisShiftChange is the spade event name for changes in shifts in aegis
	aegisShiftChange = "aegis_shift_change"
)

// List of shift actions
const (
	shiftActionOn           = "on"
	shiftActionOff          = "off"
	shiftActionChangeWeight = "change_weight"
)

// ShiftChange is a event for a change in shift
type ShiftChange struct {
	Action      string `json:"action"`
	AdminID     int64  `json:"admin_id"`
	ChangedBy   int64  `json:"changed_by"`
	Environment string `json:"environment"`
	RoleType    string `json:"role_type"`
	Weight      int64  `json:"weight"`
}

// TrackShiftOn tracks when an admin goes on shift
func (c *Client) TrackShiftOn(shift models.Shift) {
	c.spade.QueueEvents(spade.Event{
		Name: aegisShiftChange,
		Properties: ShiftChange{
			Action:      shiftActionOn,
			AdminID:     shift.AdminID,
			ChangedBy:   shift.AdminID,
			Environment: c.environment,
			RoleType:    string(shift.Type),
			Weight:      shift.Weight,
		},
	})
}

// TrackShiftChangeWeight tracks when an admin's shift weight changes
func (c *Client) TrackShiftChangeWeight(shift models.Shift, changedBy int64) {
	c.spade.QueueEvents(spade.Event{
		Name: aegisShiftChange,
		Properties: ShiftChange{
			Action:      shiftActionChangeWeight,
			AdminID:     shift.AdminID,
			ChangedBy:   changedBy,
			Environment: c.environment,
			RoleType:    string(shift.Type),
			Weight:      shift.Weight,
		},
	})
}

// TrackShiftOff tracks when an admin goes off shift
func (c *Client) TrackShiftOff(shift models.Shift, changedBy int64) {
	c.spade.QueueEvents(spade.Event{
		Name: aegisShiftChange,
		Properties: ShiftChange{
			Action:      shiftActionOff,
			AdminID:     shift.AdminID,
			ChangedBy:   changedBy,
			Environment: c.environment,
			RoleType:    string(shift.Type),
			Weight:      shift.Weight,
		},
	})
}
