import * as React from "react";
import * as ReactDOM from "react-dom";

import * as stores from "aegis/stores";
import App from "./App";

import { defaultDataIdFromObject, InMemoryCache } from "apollo-cache-inmemory";
import { ApolloClient } from "apollo-client";
import { ApolloLink } from "apollo-link";
import { setContext } from "apollo-link-context";
import { onError } from "apollo-link-error";
import { createHttpLink } from "apollo-link-http";
import { ServerError } from "apollo-link-http-common";
import { toIdValue } from "apollo-utilities";
import { ApolloProvider } from "react-apollo";
import { fragmentMatcher } from "./apollo";

import { autorun } from "mobx";
import { Provider } from "mobx-react";

import "aegis/functionality/date-extensions";
import "aegis/functionality/string-extensions";

import { gqlStorageKey } from "aegis/models/gql_request";
import "twitch-core-ui/css/index.css";
import "./index.scss";

export function initAegis() {
  const aegis = new Aegis();
  return aegis;
}

export class Aegis {
  constructor() {
    const errorLink = onError(({ networkError }) => {
      const serverError = networkError as ServerError;
      if (
        serverError &&
        (serverError.statusCode === 403 || (serverError.response && serverError.response.status === 403))
      ) {
        stores.authStore.reset(serverError.result.message);
      }
    });

    const link = createHttpLink({ uri: "/graphql" });

    const authLink = setContext((_, { headers }) => {
      return {
        headers: {
          ...headers,
          authorization: stores.authStore.hasToken ? `Bearer ${stores.authStore.token}` : null
        }
      };
    });

    const afterwareLink = new ApolloLink((operation, forward) => {
      if (!forward) {
        return null;
      }

      return forward(operation).map(response => {
        const {
          response: { headers }
        } = operation.getContext();

        if (headers) {
          const requestID = headers.get("x-request-id");

          sessionStorage.setItem(gqlStorageKey(operation.operationName), requestID);
        } else {
          throw new Error("Response header missing from backend!");
        }
        return response;
      });
    });

    const client = new ApolloClient({
      link: ApolloLink.from([errorLink, authLink, afterwareLink, link]),
      cache: new InMemoryCache({
        cacheRedirects: {
          Query: {
            user: (_, args) => toIdValue(defaultDataIdFromObject({ __typename: "User", id: args.id })!)
          }
        },
        fragmentMatcher: fragmentMatcher
      }),
      defaultOptions: {
        watchQuery: {
          fetchPolicy: "cache-and-network",
          errorPolicy: "all"
        }
      }
    });

    autorun(async () => {
      if (!stores.authStore.hasToken) {
        await client.resetStore();
      }
    });

    ReactDOM.render(
      <ApolloProvider client={client}>
        <Provider {...stores}>
          <App />
        </Provider>
      </ApolloProvider>,
      document.getElementById("root") as HTMLElement
    );
  }
}
