import * as React from "react";

import * as QueryString from "query-string";
import { RouteComponentProps, withRouter } from "react-router";

import { getSearchParam } from "aegis/functionality/utils/contrib/twilight/url-helpers";
import { Layout } from "twitch-core-ui";

import { DetailedUser } from "aegis/models";
import { QueryProps } from "aegis/pages/account-info";
import { AccountInfoChannelStreamInfo } from "./channel-stream-info";
import { AccountInfoEnforcementHistory } from "./enforcement-history";
import { AccountInfoProfile } from "./profile";
import { AccountInfoTabs, InfoTab } from "./tabs";

export interface PublicProps {
  user: DetailedUser;
}

type RouteProps = RouteComponentProps<{}>;

export type Props = PublicProps & RouteProps;

export enum TestSelectors {
  PopulatedBody = "account-info-populated-body",
  BodyContent = "account-info-body-content"
}

export class AccountInfoBodyComponent extends React.Component<Props> {
  public constructor(props: Props) {
    super(props);
    if (!this.getCurrentTab()) {
      this.changeTab(InfoTab.Profile);
    }
  }

  public render() {
    const { user } = this.props;
    const currentTab = this.getCurrentTab();
    if (!currentTab) {
      return null;
    }

    return (
      <Layout flexGrow={1} data-test-selector={TestSelectors.PopulatedBody}>
        {currentTab ? <AccountInfoTabs currentTab={currentTab} changeTab={this.changeTab} /> : null}
        <AccountInfoBodyContent user={user} currentTab={currentTab} data-test-selector={TestSelectors.BodyContent} />
      </Layout>
    );
  }

  private changeTab = (tab: InfoTab) => {
    const queryParams = QueryString.parse(location.search) as QueryProps;
    const updatedParams = QueryString.stringify({ ...queryParams, tabID: tab });
    this.props.history.push({ pathname: "/accountinfo", search: updatedParams });
  };

  private getCurrentTab = () => {
    const tabID = getSearchParam(this.props.location.search, "tabID");
    if (!tabID || !(tabID in InfoTab)) {
      return null;
    }
    return tabID as InfoTab;
  };
}

export const AccountInfoBody: React.ComponentClass<PublicProps> = withRouter<Props>(AccountInfoBodyComponent);

interface ContentProps {
  currentTab: InfoTab;
}

export const AccountInfoBodyContent: React.SFC<PublicProps & ContentProps> = ({ currentTab, user }) => {
  switch (currentTab) {
    case InfoTab.Profile:
      return <AccountInfoProfile user={user} />;
    case InfoTab.ChannelStreamInfo:
      return <AccountInfoChannelStreamInfo user={user} />;
    case InfoTab.EnforcementHistory:
      return <AccountInfoEnforcementHistory id={user.id} />;
    default:
      return null;
  }
};
