import * as React from "react";

import gql from "graphql-tag";

import { LoadingFill } from "aegis/features/loading-fill";
import { PanelFields, UserBadgesFields } from "aegis/fragments";
import { UserEmoticonFields } from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { DetailedUser } from "aegis/models";
import { Color, CoreText, Layout } from "twitch-core-ui";
import { AccountInfoProfileComponent } from "./component";
import { ProfileData } from "./models";

export const USER_PROFILE_QUERY = gql`
  query UserProfileQuery($id: ID!) {
    profileData: user(id: $id) {
      id
      chatRules
      panels {
        ...PanelFields
      }
      emoticons {
        ...UserEmoticonFields
      }
      badges {
        ...UserBadgesFields
      }
      profileImage
      description
      channel {
        offlineImage
        profileBannerImage
      }
    }
  }

  ${PanelFields}
  ${UserEmoticonFields}
  ${UserBadgesFields}
`;

interface Props {
  user: DetailedUser;
}

type UserProfileQueryVariables = {
  id: string;
};

interface UserProfileQueryResponse {
  profileData?: ProfileData;
}

export class UserProfileQuery extends Query<UserProfileQueryResponse, UserProfileQueryVariables> {}

export class AccountInfoProfile extends React.PureComponent<Props> {
  public render() {
    const { user } = this.props;
    return (
      <UserProfileQuery displayName="AccountInfoProfile" query={USER_PROFILE_QUERY} variables={{ id: user.id }}>
        {({ data, error, loading }) => {
          if (loading) {
            return <LoadingFill />;
          } else if (error) {
            console.error("An error occurred while fetching detailed panel info", error);
            return <CoreText color={Color.Error}>An error occurred fetching additional details for {user.id}</CoreText>;
          } else if (!data || !data.profileData) {
            return (
              <Layout>
                <CoreText color={Color.Error}>No profile data found for user {user.id}</CoreText>
              </Layout>
            );
          } else {
            return <AccountInfoProfileComponent user={user} profileData={data.profileData} />;
          }
        }}
      </UserProfileQuery>
    );
  }
}
