import * as React from "react";

import { HeaderTitle } from "aegis/functionality/components/header-title";
import { IndicatorButton } from "aegis/functionality/components/indicator-button/component";
import { channelLink, DetailedUser, Panel } from "aegis/models";
import {
  ButtonType,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FlexWrap,
  JustifyContent,
  Layout,
  StyledLayout,
  SVGAsset,
  TextType,
  WordBreak
} from "twitch-core-ui";

import "./styles.scss";

export interface Props {
  user: DetailedUser;
  panels?: Panel[];
}
const PanelImageWidth = 320; // Image width specified in twitch site

export enum TestSelector {
  Panel = "account-info-profile-panel--panel",
  PanelImage = "account-info-profile-panel--image",
  DashboardButton = "account-info-profile-panel--dashboard"
}

export const Panels: React.SFC<Props> = ({ user, panels }) => {
  const panelsComponents =
    !panels || panels.length === 0 ? (
      <CoreText type={TextType.Em} color={Color.Alt2}>
        No panels
      </CoreText>
    ) : (
      panels.map((p, i) => {
        return (
          <StyledLayout
            key={i}
            flexShrink={0}
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            flexWrap={FlexWrap.Wrap}
            margin={{ right: 1, bottom: 1 }}
            padding={0.5}
            elevation={3}
            data-test-selector={TestSelector.Panel}
            className="panel"
          >
            <div>
              <CoreText wordBreak={WordBreak.BreakAll}>
                <strong>ID</strong>: {p.id}
              </CoreText>
              <CoreText wordBreak={WordBreak.BreakAll}>
                <strong>Title</strong>: {p.title}
              </CoreText>
              <CoreText wordBreak={WordBreak.BreakAll}>
                <strong>Description</strong>: {p.description}
              </CoreText>
              <CoreText wordBreak={WordBreak.BreakAll}>
                <strong>Image Link To</strong>: {p.linkTo}
              </CoreText>
              <CoreText wordBreak={WordBreak.BreakAll}>
                <strong>Image URL</strong>: {p.imageURL}
              </CoreText>
            </div>
            <div>
              {p.imageURL && (
                <img src={p.imageURL} width={PanelImageWidth} data-test-selector={TestSelector.PanelImage} />
              )}
            </div>
          </StyledLayout>
        );
      })
    );

  return (
    <Layout fullWidth display={Display.Flex} flexDirection={FlexDirection.Column}>
      <HeaderTitle title="Panels" />
      {panels &&
        panels.length > 0 && (
          <Layout fullWidth display={Display.Flex} justifyContent={JustifyContent.End}>
            <div>
              <IndicatorButton
                indicatorIcon={SVGAsset.LogoGlitch}
                type={ButtonType.Text}
                targetBlank
                linkTo={channelLink(user)}
                data-test-selector={TestSelector.DashboardButton}
              >
                Manage in channel
              </IndicatorButton>
            </div>
          </Layout>
        )}
      <Layout padding={1} display={Display.Flex} flexWrap={FlexWrap.Wrap}>
        {panelsComponents}
      </Layout>
    </Layout>
  );
};
