import * as React from "react";

import { createContext } from "aegis/functionality/utils/contrib/twilight/context";
import { DetailedUser } from "aegis/models";

export enum DeletableTypes {
  ChatRule = "chat-rule",
  Panel = "panel",
  Emotes = "emotes",
  ChannelImages = "channel-images",
  Description = "description",
  Badges = "badges"
}

export interface AccountInfoDeleteContext {
  toggleMarkForDeletion: (contentType: DeletableTypes, contentID: string) => void;
  markForDeletion: (contentType: DeletableTypes, contentIDs: string[]) => void;
  unmarkForDeletion: (contentType: DeletableTypes, contentIDs: string[]) => void;
  isMarked: (contentType: DeletableTypes, contentIDs: string[]) => boolean;
  showDeletionBar: () => boolean;
  cancelSelection: () => void;
  getMarkedForDeletion: () => Map<DeletableTypes, Set<string>>;
  getNumMarkedElements: () => number;
}

const mutationsInitValues: AccountInfoDeleteContext = {
  toggleMarkForDeletion: () => undefined,
  markForDeletion: () => undefined,
  unmarkForDeletion: () => undefined,
  isMarked: () => false,
  showDeletionBar: () => false,
  cancelSelection: () => undefined,
  getMarkedForDeletion: () => new Map(),
  getNumMarkedElements: () => 0
};

export interface DeleteContextState {
  markedForDeletion: Map<DeletableTypes, Set<string>>;
}

export const {
  InnerProvider: AccountInfoDeleteInnerProvider,
  withContext: withAccountInfoDeleteContext,
  FullContext: AccountInfoDeleteFullContext
} = createContext<AccountInfoDeleteContext>("AccountInfoDeleteContext", mutationsInitValues);

export interface PublicProps {
  user: DetailedUser;
}

export class AccountInfoDeleteContextProvider extends React.Component<PublicProps, DeleteContextState> {
  public state = {
    markedForDeletion: new Map<DeletableTypes, Set<string>>()
  };

  public render() {
    return (
      <AccountInfoDeleteInnerProvider
        value={{
          toggleMarkForDeletion: this.toggleMarkForDeletion,
          isMarked: this.isMarked,
          markForDeletion: this.markForDeletion,
          unmarkForDeletion: this.unmarkForDeletion,
          showDeletionBar: this.showDeletionBar,
          cancelSelection: this.cancelSelection,
          getMarkedForDeletion: this.getMarkedForDeletion,
          getNumMarkedElements: this.getNumMarkedElements
        }}
      >
        {this.props.children}
      </AccountInfoDeleteInnerProvider>
    );
  }

  private getMarkedForDeletion = () => {
    return this.state.markedForDeletion;
  };

  private isMarked = (contentType: DeletableTypes, contentIDs: string[]) => {
    const contentOfType = this.state.markedForDeletion.get(contentType);
    if (!contentOfType) {
      return false;
    }
    let included = true;
    contentIDs.forEach(contentID => {
      if (!contentOfType.has(contentID)) {
        included = false;
      }
    });
    return included;
  };

  private toggleMarkForDeletion = (contentType: DeletableTypes, contentID: string) => {
    if (this.isMarked(contentType, [contentID])) {
      this.unmarkForDeletion(contentType, [contentID]);
    } else {
      this.markForDeletion(contentType, [contentID]);
    }
  };

  private markForDeletion = (contentType: DeletableTypes, contentIDs: string[]) => {
    if (!this.isMarked(contentType, contentIDs)) {
      this.setState(state => {
        const contentOfType = state.markedForDeletion.get(contentType) || new Set();
        contentIDs.forEach(id => {
          contentOfType.add(id);
        });
        state.markedForDeletion.set(contentType, contentOfType);

        return { markedForDeletion: state.markedForDeletion };
      });
    }
  };

  private cancelSelection = () => {
    this.setState({ markedForDeletion: new Map<DeletableTypes, Set<string>>() });
  };

  private unmarkForDeletion = (contentType: DeletableTypes, contentIDs: string[]) => {
    if (this.isMarked(contentType, contentIDs)) {
      this.setState(state => {
        const contentOfType = state.markedForDeletion.get(contentType) || new Set();
        contentIDs.forEach(id => {
          contentOfType.delete(id);
        });
        if (contentOfType.size > 0) {
          state.markedForDeletion.set(contentType, contentOfType);
        } else {
          state.markedForDeletion.delete(contentType);
        }

        return { markedForDeletion: state.markedForDeletion };
      });
    }
  };

  private showDeletionBar = () => {
    return this.state.markedForDeletion.size > 0;
  };

  private getNumMarkedElements = () => {
    let count = 0;
    this.state.markedForDeletion.forEach(v => {
      count = count + v.size;
    });
    return count;
  };
}

export function accountInfoDeletionContextMapper(context: AccountInfoDeleteContext): AccountInfoDeleteContext {
  return context;
}
