import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { DetailedUser } from "aegis/models";
import { MockDetailedUser } from "aegis/models/mocks";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { Button } from "twitch-core-ui";
import { AccountInfoDeleteContextProvider } from "../../../common/delete-content";
import { DeletableTypes } from "../context";
import { TestSelectors } from "./component";
import { DeleteBar, DeleteBarContainer } from "./container";
import { FETCH_BADGE_REASONS } from "./deletion-reasons";

Enzyme.configure({ adapter: new Adapter() });

const setupMount = (userOverride?: DetailedUser) => {
  const user = userOverride ? userOverride : MockDetailedUser();
  const apolloMocks = [
    {
      request: {
        query: FETCH_BADGE_REASONS,
        variables: {}
      },
      result: {
        data: {}
      }
    }
  ];

  return Enzyme.mount(
    wrapWithGQLProvider(
      <AccountInfoDeleteContextProvider user={user}>
        <DeleteBar user={user} />
      </AccountInfoDeleteContextProvider>,
      apolloMocks
    ).wrappedComponent
  );
};

describe("DeleteBar", () => {
  it("does not render when empty", () => {
    const wrapper = setupMount();
    expect(wrapper.find({ "data-test-selector": TestSelectors.DeleteBar })).toEqual({});
  });

  it("renders when selection is made", () => {
    const wrapper = setupMount();
    wrapper
      .find(DeleteBarContainer)
      .props()
      .markForDeletion(DeletableTypes.Emotes, ["fake-emote-id"]);
    wrapper.update();

    expect(wrapper.find({ "data-test-selector": TestSelectors.DeleteBar })).toBeDefined();
  });

  it("renders count correctly", () => {
    const wrapper = setupMount();
    wrapper
      .find(DeleteBarContainer)
      .props()
      .markForDeletion(DeletableTypes.Emotes, ["fake-emote-id"]);
    wrapper
      .find(DeleteBarContainer)
      .props()
      .markForDeletion(DeletableTypes.ChatRule, ["all"]);
    wrapper.update();

    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.DeleteButton })
        .last()
        .text()
    ).toBe("Delete Selected (2)");
  });

  it("disables delete button when badge selected with no reason", () => {
    const wrapper = setupMount();
    wrapper
      .find(DeleteBarContainer)
      .props()
      .markForDeletion(DeletableTypes.Badges, ["fake-badge-id"]);
    wrapper.update();

    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.DeleteButton })
        .find(Button)
        .props().disabled
    ).toBeTruthy();
  });

  it("cancels selection and hides on cancel clicked", () => {
    const wrapper = setupMount();
    wrapper
      .find(DeleteBarContainer)
      .props()
      .markForDeletion(DeletableTypes.Emotes, ["fake-emote-id"]);
    wrapper.update();

    wrapper
      .find(DeleteBarContainer)
      .first()
      .simulate("click");

    wrapper.update();

    expect(wrapper.find({ "data-test-selector": TestSelectors.DeleteBar })).toEqual({});
  });
});
