import * as React from "react";

import gql from "graphql-tag";
import { Mutation } from "react-apollo";

import { DetailedUserFields, UserFields } from "aegis/fragments";
import { apolloErrorHandler } from "aegis/functionality/error-notifications";
import { DetailedUser } from "aegis/models";
import { USER_PROFILE_QUERY } from "../../../body/profile";
import { ChannelImageTypes } from "../../../body/profile/channel-images/component";
import {
  AccountInfoDeleteContext,
  accountInfoDeletionContextMapper,
  DeletableTypes,
  withAccountInfoDeleteContext
} from "../context";
import { DeleteBarComponent } from "./component";

export const DELETE_USER_CONTENT = gql`
  mutation DeleteUserContent($input: DeleteUserContentInput!) {
    deleteUserContent(input: $input) {
      user {
        ...UserFields
        ...DetailedUserFields
      }
    }
  }
  ${UserFields}
  ${DetailedUserFields}
`;

type DeleteUserContentInput = {
  input: {
    userID: string;
    profileImage?: boolean;
    profileBanner?: boolean;
    offlineImage?: boolean;
    description?: boolean;
    chatRules?: boolean;
    emoticonIDs?: string[];
    badges?: {
      reason: string;
      badgeIDs: string[];
    };
  };
};

type DeleteUserContentResponse = {
  deleteUserContent: {
    user: DetailedUser;
  };
};

export class DeleteUserContentMutation extends Mutation<DeleteUserContentResponse, DeleteUserContentInput> {}

export interface DeleteBarPublicProps {
  user: DetailedUser;
}

interface State {
  reason?: string;
}

export class DeleteBarContainer extends React.Component<DeleteBarPublicProps & AccountInfoDeleteContext, State> {
  state: State = {};

  public render() {
    if (this.props.showDeletionBar()) {
      return (
        <DeleteUserContentMutation
          mutation={DELETE_USER_CONTENT}
          refetchQueries={[{ query: USER_PROFILE_QUERY }]}
          onError={apolloErrorHandler("Failed to delete all content")}
        >
          {deleteContentMutation => {
            const markedForDeletion = this.props.getMarkedForDeletion();
            const deleteContent = () => {
              const { user } = this.props;
              const markedImages = markedForDeletion.get(DeletableTypes.ChannelImages);
              const markedEmotes = markedForDeletion.get(DeletableTypes.Emotes);
              const markedBadges = markedForDeletion.get(DeletableTypes.Badges);

              const input: DeleteUserContentInput = {
                input: {
                  userID: user.id,
                  profileImage: markedImages && !!markedImages.has(ChannelImageTypes.ProfileImage),
                  profileBanner: markedImages && !!markedImages.has(ChannelImageTypes.ProfileBanner),
                  offlineImage: markedImages && !!markedImages.has(ChannelImageTypes.OfflineImage),
                  description: !!markedForDeletion.get(DeletableTypes.Description),
                  chatRules: !!markedForDeletion.get(DeletableTypes.Description),
                  emoticonIDs: markedEmotes && Array.from(markedEmotes),
                  badges:
                    markedBadges && this.state.reason
                      ? {
                          badgeIDs: Array.from(markedBadges),
                          reason: this.state.reason
                        }
                      : undefined
                }
              };

              deleteContentMutation({ variables: input });

              this.props.cancelSelection();
            };
            return (
              <DeleteBarComponent
                cancelSelection={this.props.cancelSelection}
                deleteContent={deleteContent}
                showReasonSelector={!!markedForDeletion.get(DeletableTypes.Badges)}
                deleteDisabled={
                  !!markedForDeletion.get(DeletableTypes.Badges) && (!this.state.reason || this.state.reason === "")
                }
                selectDeletionReason={(reason: string) => {
                  this.setState({ reason: reason });
                }}
                selectedElementCount={this.props.getNumMarkedElements()}
                user={this.props.user}
              />
            );
          }}
        </DeleteUserContentMutation>
      );
    }
    return null;
  }
}

export const DeleteBar = withAccountInfoDeleteContext<AccountInfoDeleteContext, DeleteBarPublicProps>(
  accountInfoDeletionContextMapper
)(DeleteBarContainer);
