import * as React from "react";

import { WithTranslatePopup } from "aegis/features/translate";
import { TranslateIconPlacement } from "aegis/features/translate/component";
import { UserCardLink } from "aegis/features/user-card/user-link";
import { bullet, DetailedUser } from "aegis/models";

import {
  Avatar,
  BorderRadius,
  Color,
  CoreLink,
  CoreText,
  Display,
  FontSize,
  Layout,
  Placeholder,
  StyledLayout,
  TextType,
  Tooltip
} from "twitch-core-ui";

import "./styles.scss";

export interface PublicProps {
  user: DetailedUser;
}

export class AccountInfoHeaderBanner extends React.Component<PublicProps> {
  public render() {
    let profileImage = null;
    let channelLangText = null;
    let accountLangText = <Placeholder />;
    let onlineStatus = <Placeholder />;
    let displayNameText = <Placeholder />;
    let offensiveLevelText = <Placeholder />;
    let loginText = null;

    const { user } = this.props;
    if (user.channel) {
      if (user.channel.language) {
        channelLangText = (
          <Layout display={Display.InlineBlock}>
            <CoreText key="separator" type={TextType.Span}>
              {bullet}{" "}
            </CoreText>
            <span key="accountLang" className="semi-bold tw-upcase">
              {user.channel.language}
            </span>
            <span key="label" className="semi-thin">
              {" channel"}
            </span>
          </Layout>
        );
      }
    }
    profileImage = user.profileImage;
    accountLangText = (
      <Layout display={Display.InlineBlock} margin={{ right: 0.5 }}>
        <span key="accountLang" className="semi-bold tw-upcase">
          {user.language}
        </span>
        <span key="label" className="semi-thin">
          {" account"}
        </span>
      </Layout>
    );

    let displayNameLabel: string = "display name";
    if (user.displayName.toLowerCase() !== user.login.toLowerCase()) {
      loginText = (
        <Layout margin={{ left: 0.5 }} display={Display.Inline}>
          {" ("}
          <CoreText type={TextType.Span}>{user.login}</CoreText>
          <WithTranslatePopup text={user.login} iconPlacement={TranslateIconPlacement.After} label="username">
            <CoreText />
          </WithTranslatePopup>
          {")"}
        </Layout>
      );
    } else {
      displayNameLabel = "username";
    }
    displayNameText = (
      <WithTranslatePopup text={user.displayName} iconPlacement={TranslateIconPlacement.After} label={displayNameLabel}>
        <CoreText type={TextType.Span}>{user.displayName}</CoreText>
      </WithTranslatePopup>
    );

    onlineStatus = <CoreText type={TextType.Span}>{user.isLiveStreaming ? "Live" : "Not Live"}</CoreText>;
    offensiveLevelText = (
      <Tooltip label="Username offensive level">
        <CoreText
          type={TextType.Span}
          color={user.acceptableNameConfidence! < 0.8 ? Color.Overlay : Color.Error}
          fontSize={FontSize.Size8}
        >
          {`(${user.acceptableNameConfidence!.toFixed(2)})`}
        </CoreText>
      </Tooltip>
    );

    return (
      <StyledLayout padding={{ left: 1, right: 2, y: 1 }} color={Color.Alt2} borderRight>
        <Layout display={Display.Flex}>
          <div className="avatar-container">
            <UserCardLink userID={user.id}>
              <CoreLink>
                <Avatar size={36} src={profileImage} alt="Avatar" borderRadius={BorderRadius.None} />
              </CoreLink>
            </UserCardLink>
          </div>
          <StyledLayout color={Color.Base} display={Display.Block}>
            <CoreText type={TextType.Span}>
              <span className={"semi-bold"}>{displayNameText}</span>
              <span className={"semi-thin"}>
                {loginText} {offensiveLevelText}
              </span>
            </CoreText>
            <CoreText fontSize={FontSize.Size8} color={Color.Alt2}>
              {onlineStatus}
            </CoreText>
          </StyledLayout>
        </Layout>
        <StyledLayout className="lang-container" display={Display.Flex} fontSize={FontSize.Size7}>
          <Layout flexOrder={0}>
            {accountLangText}
            {channelLangText}
          </Layout>
        </StyledLayout>
      </StyledLayout>
    );
  }
}
