import * as React from "react";

import gql from "graphql-tag";

import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { StackedModal } from "aegis/functionality/components/stacked-modal";
import { Admin, DetailedUser } from "aegis/models";
import { AdminActionsModalComponent } from "./component";

export interface PublicProps {
  user: DetailedUser;
  closeModal: () => void;
}

export const CONTENT_AUDIT_LIMIT = 10;

export const FETCH_CONTENT_AUDITS = gql`
  query contentAudits($id: ID, $offset: Int) {
    user(id: $id) {
      id
      contentAudits(limit: ${CONTENT_AUDIT_LIMIT}, offset: $offset) {
        audits {
          id
          action
          actionBy {
            ...AdminFields
          }
          resourceID
          resourceType
          createdAt
        }
        pageInfo {
          total
          offset
        }
      }
    }
  }
  ${Fragments.AdminFields}
`;

type Variables = {
  id: string;
  offset: number;
};

interface Data {
  user: {
    contentAudits?: {
      audits: ContentAudit[];
      pageInfo: {
        total: number;
        offset: number;
      };
    };
  };
}

export interface ContentAudit {
  id: string;
  action: string;
  actionBy: Admin;
  resourceID: string;
  resourceType: string;
  createdAt: string;
}

class ContentAuditsQuery extends Query<Data, Variables> {}

export const AdminActionsModal: React.SFC<PublicProps> = ({ user, closeModal }) => {
  return (
    <StackedModal onClose={closeModal} title={`Admin actions for ${user.displayName}`}>
      <ContentAuditsQuery
        displayName="AdminActionsModal"
        query={FETCH_CONTENT_AUDITS}
        variables={{ id: user.id, offset: 0 }}
      >
        {({ loading, error, data, refetch }) => {
          if (error) {
            console.error(error);
            return null;
          }

          const contentAudits = data && data.user ? data.user.contentAudits : undefined;
          const getAuditsForPage = (pageNumber: number) => {
            refetch({ id: user.id, offset: CONTENT_AUDIT_LIMIT * (pageNumber - 1) });
          };
          const total = contentAudits ? contentAudits.pageInfo.total : 0;
          const offset = contentAudits ? contentAudits.pageInfo.offset : 0;

          return (
            <AdminActionsModalComponent
              adminActions={contentAudits ? contentAudits.audits : []}
              getAuditsForPage={getAuditsForPage}
              total={total}
              offset={offset}
              loading={loading}
            />
          );
        }}
      </ContentAuditsQuery>
    </StackedModal>
  );
};
