import * as React from "react";

import gql from "graphql-tag";

import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { Admin, DetailedUser } from "aegis/models";
import { LoadingSpinner } from "twitch-core-ui";
import { AdminActionsComponent } from "./component";

export interface PublicProps {
  user: DetailedUser;
}

export const FETCH_CONTENT_AUDITS = gql`
  query contentAudits($id: ID) {
    user(id: $id) {
      id
      contentAudits(limit: 10, offset: 0) {
        audits {
          id
          action
          actionBy {
            ...AdminFields
          }
          resourceID
          resourceType
          createdAt
        }
        pageInfo {
          total
        }
      }
    }
  }
  ${Fragments.AdminFields}
`;

type Variables = {
  id: string;
};

interface Data {
  user: {
    contentAudits?: {
      audits: ContentAudit[];
      pageInfo: {
        total: Number;
      };
    };
  };
}

export interface ContentAudit {
  id: string;
  action: string;
  actionBy: Admin;
  resourceID: string;
  resourceType: string;
  createdAt: string;
}

class ContentAuditsQuery extends Query<Data, Variables> {}

export const AdminActions: React.SFC<PublicProps> = ({ user }) => {
  return (
    <ContentAuditsQuery displayName="AdminActions" query={FETCH_CONTENT_AUDITS} variables={{ id: user.id }}>
      {({ loading, error, data }) => {
        if (error || loading || !data) {
          return <LoadingSpinner />;
        }
        const { contentAudits } = data.user;
        const showMore = !!contentAudits && contentAudits.pageInfo.total > contentAudits.audits.length;
        return (
          <AdminActionsComponent
            user={user}
            adminActions={contentAudits ? contentAudits.audits : []}
            showMore={showMore}
          />
        );
      }}
    </ContentAuditsQuery>
  );
};
