import * as React from "react";

import { AlignItems, CoreLink, CoreText, Display, FlexDirection, Layout, SVG, SVGAsset } from "twitch-core-ui";

import { SuspensionLabel } from "aegis/features/report-view/components/report-body/user-metadata/user-metadata";
import { UserRoleTag } from "aegis/features/user-role-tags";
import { HeaderTitle } from "aegis/functionality/components/header-title";
import { DetailedUser } from "aegis/models";
import { UserRoleTagMapping } from "aegis/models";
import { AdminActions } from "./admin-actions";
import { PartnerEscalation } from "./partner-escalation";
import { StrikeDetails } from "./strike-details";

export interface PublicProps {
  user: DetailedUser;
}

export const AccountInfoSidebar: React.SFC<PublicProps> = ({ user }) => {
  return (
    <Layout fullWidth display={Display.Flex} flexDirection={FlexDirection.Column} padding={{ top: 4, x: 1 }}>
      <HeaderTitle title="Account state" />
      <Layout padding={{ left: 1, top: 1, bottom: 2 }}>
        <AccountState user={user} />
        <Layout padding={{ top: 1 }} />
        <StrikeDetails user={user} />
      </Layout>
      <HeaderTitle title="Account type" />
      <Layout padding={{ left: 1, top: 1, bottom: 1 }}>
        <AccountRoles user={user} />
        <PartnerEscalation user={user} />
      </Layout>
      <HeaderTitle title="Stream title history" />
      <Layout padding={{ left: 1, top: 1, bottom: 2 }}>
        <CoreLink
          linkTo={`https://www.twitch.tv/${encodeURIComponent(user.login)}/dashboard/activity?tmi_host=false`}
          targetBlank
        >
          <Layout display={Display.InlineFlex} alignItems={AlignItems.Center}>
            <Layout margin={{ right: 0.5 }}>View in channel dashboard</Layout>
            <SVG asset={SVGAsset.LogoGlitch} />
          </Layout>
        </CoreLink>
      </Layout>
      <HeaderTitle title="Admin actions" />
      <Layout padding={{ left: 1, top: 1, bottom: 2 }}>
        <AdminActions user={user} />
      </Layout>
    </Layout>
  );
};

export const AccountState: React.SFC<{ user: DetailedUser }> = ({ user }) => {
  if (user.suspension) {
    return <SuspensionLabel user={user} />;
  } else if (user.isBanned || user.isDMCABanned) {
    return <CoreText>Banned indefinitely, suspension not found</CoreText>;
  }
  return <UserRoleTag text="Active" icon={SVGAsset.PermissionsAdded} />;
};

const AccountRoles: React.SFC<{ user: DetailedUser }> = ({ user }) => {
  if (!user.roles || user.roles.length === 0) {
    return <Layout padding={{ bottom: 1 }}>No roles on this user</Layout>;
  }
  const roles = user.roles.map((userRole, index) => {
    const roleValue = UserRoleTagMapping.get(userRole);
    if (roleValue) {
      return (
        <Layout padding={{ bottom: 1 }} key={"role-tag-" + index}>
          <UserRoleTag {...roleValue} />
        </Layout>
      );
    } else {
      console.warn("Unrecognized user role: ", userRole);
      return (
        <Layout padding={{ bottom: 1 }}>
          <UserRoleTag text={userRole} icon={SVGAsset.QuestionMark} />
        </Layout>
      );
    }
  });
  return <>{roles}</>;
};
