import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { MockDetailedUser } from "aegis/models/mocks";
import { StrikeDetailsComponent, TestSelector } from "./component";

import { ConfirmationModalConfirmSelector } from "aegis/functionality/components/confirmation-modal/component";
import "aegis/functionality/date-extensions";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { Button } from "twitch-core-ui";

Enzyme.configure({ adapter: new Adapter() });

describe("StrikeDetails", () => {
  const modalRoot = document.createElement("div");
  modalRoot.setAttribute("id", "portal-modal-root");
  const body = document.querySelector("body");
  if (body) {
    body.appendChild(modalRoot);
  }

  const globalDiv = document.createElement("div");
  document.body.appendChild(globalDiv);

  it("renders when user is not banned and has no strike count", () => {
    const wrapper = Enzyme.mount(
      <StrikeDetailsComponent
        user={MockDetailedUser({
          isBanned: false,
          isDMCABanned: false,
          tosViolationCount: 0,
          dmcaViolationCount: 0
        })}
        onUnban={jest.fn()}
      />
    );

    expect(
      wrapper
        .find({ "data-test-selector": TestSelector.TOSViolationCountText })
        .hostNodes()
        .text()
    ).toBe("0 TOS strikes");

    expect(
      wrapper
        .find({ "data-test-selector": TestSelector.DMCAViolationCountText })
        .hostNodes()
        .text()
    ).toBe("0 DMCA strikes");

    expect(wrapper.find({ "data-test-selector": TestSelector.DecrementTOSCheckbox })).toHaveLength(0);
    expect(wrapper.find({ "data-test-selector": TestSelector.DecrementDMCACheckbox })).toHaveLength(0);
    expect(wrapper.find({ "data-test-selector": TestSelector.UnbanButton })).toHaveLength(0);
  });

  it("renders when user is currently tos banned", () => {
    const wrapper = Enzyme.mount(
      <StrikeDetailsComponent
        user={MockDetailedUser({
          isBanned: true,
          isDMCABanned: false,
          tosViolationCount: 1,
          dmcaViolationCount: 0
        })}
        onUnban={jest.fn()}
      />
    );

    expect(
      wrapper
        .find({ "data-test-selector": TestSelector.TOSViolationCountText })
        .hostNodes()
        .text()
    ).toBe("1 TOS strikes");

    expect(
      wrapper
        .find({ "data-test-selector": TestSelector.DMCAViolationCountText })
        .hostNodes()
        .text()
    ).toBe("0 DMCA strikes");

    expect(wrapper.find({ "data-test-selector": TestSelector.DecrementTOSCheckbox }).hostNodes()).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelector.DecrementDMCACheckbox })).toHaveLength(0);
    expect(wrapper.find({ "data-test-selector": TestSelector.UnbanButton }).hostNodes()).toHaveLength(1);
  });

  it("renders when user is currently dmca banned", () => {
    const wrapper = Enzyme.mount(
      <StrikeDetailsComponent
        user={MockDetailedUser({
          isBanned: false,
          isDMCABanned: true,
          tosViolationCount: 0,
          dmcaViolationCount: 1
        })}
        onUnban={jest.fn()}
      />
    );

    expect(
      wrapper
        .find({ "data-test-selector": TestSelector.DMCAViolationCountText })
        .hostNodes()
        .text()
    ).toBe("1 DMCA strikes");

    expect(
      wrapper
        .find({ "data-test-selector": TestSelector.TOSViolationCountText })
        .hostNodes()
        .text()
    ).toBe("0 TOS strikes");

    expect(wrapper.find({ "data-test-selector": TestSelector.DecrementTOSCheckbox })).toHaveLength(0);
    expect(wrapper.find({ "data-test-selector": TestSelector.DecrementDMCACheckbox }).hostNodes()).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelector.UnbanButton }).hostNodes()).toHaveLength(1);
  });

  it("renders when user is both tos and dmca banned", () => {
    const wrapper = Enzyme.mount(
      <StrikeDetailsComponent
        user={MockDetailedUser({
          isBanned: true,
          isDMCABanned: true,
          tosViolationCount: 1,
          dmcaViolationCount: 2
        })}
        onUnban={jest.fn()}
      />
    );

    expect(
      wrapper
        .find({ "data-test-selector": TestSelector.TOSViolationCountText })
        .hostNodes()
        .text()
    ).toBe("1 TOS strikes");

    expect(
      wrapper
        .find({ "data-test-selector": TestSelector.DMCAViolationCountText })
        .hostNodes()
        .text()
    ).toBe("2 DMCA strikes");

    expect(wrapper.find({ "data-test-selector": TestSelector.DecrementTOSCheckbox }).hostNodes()).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelector.DecrementDMCACheckbox }).hostNodes()).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelector.UnbanButton }).hostNodes()).toHaveLength(1);
  });

  it("triggers onBan callback on confirmation", () => {
    const fn = jest.fn();
    const wrapper = Enzyme.mount(
      wrapWithGQLProvider(
        <StrikeDetailsComponent
          user={MockDetailedUser({
            isBanned: true,
            isDMCABanned: true,
            tosViolationCount: 1,
            dmcaViolationCount: 1
          })}
          onUnban={fn}
        />,
        []
      ).wrappedComponent
    );

    expect(
      wrapper
        .find({ "data-test-selector": TestSelector.DMCAViolationCountText })
        .hostNodes()
        .text()
    ).toBe("1 DMCA strikes");

    expect(
      wrapper
        .find({ "data-test-selector": TestSelector.TOSViolationCountText })
        .hostNodes()
        .text()
    ).toBe("1 TOS strikes");

    const decrementTOS = wrapper.find({ "data-test-selector": TestSelector.DecrementTOSCheckbox }).hostNodes();
    const decrementDMCA = wrapper.find({ "data-test-selector": TestSelector.DecrementDMCACheckbox }).hostNodes();
    const unban = wrapper.find({ "data-test-selector": TestSelector.UnbanButton }).hostNodes();
    expect(decrementTOS).toHaveLength(1);
    expect(decrementDMCA).toHaveLength(1);
    expect(unban).toHaveLength(1);

    unban.simulate("click");

    expect(fn.mock.calls).toHaveLength(0);

    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");

    expect(fn.mock.calls).toHaveLength(1);
  });

  it("triggers onBan when user is not banned but has tos violation count", () => {
    const fn = jest.fn();
    const wrapper = Enzyme.mount(
      wrapWithGQLProvider(
        <StrikeDetailsComponent
          user={MockDetailedUser({
            isBanned: false,
            isDMCABanned: false,
            tosViolationCount: 1,
            dmcaViolationCount: 0
          })}
          onUnban={fn}
        />,
        []
      ).wrappedComponent
    );

    expect(
      wrapper
        .find({ "data-test-selector": TestSelector.DMCAViolationCountText })
        .hostNodes()
        .text()
    ).toBe("0 DMCA strikes");

    expect(
      wrapper
        .find({ "data-test-selector": TestSelector.TOSViolationCountText })
        .hostNodes()
        .text()
    ).toBe("1 TOS strikes");

    const decrementTOS = wrapper.find({ "data-test-selector": TestSelector.DecrementTOSCheckbox }).hostNodes();
    const decrementDMCA = wrapper.find({ "data-test-selector": TestSelector.DecrementDMCACheckbox }).hostNodes();
    const unban = wrapper.find({ "data-test-selector": TestSelector.UnbanButton }).hostNodes();
    expect(decrementTOS).toHaveLength(1);
    expect(decrementDMCA).toHaveLength(0);
    expect(unban).toHaveLength(1);

    // Checking the input this way since simulate doesn't work
    const input = decrementTOS.find("input").getDOMNode() as HTMLInputElement;
    input.checked = true;
    decrementTOS.find("input").simulate("change", { target: { checked: true } });

    unban.simulate("click");

    expect(fn.mock.calls).toHaveLength(0);
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");

    expect(fn.mock.calls).toHaveLength(1);
  });

  it("triggers onBan when user is not banned but has dmca violation count", () => {
    const fn = jest.fn();
    const wrapper = Enzyme.mount(
      wrapWithGQLProvider(
        <StrikeDetailsComponent
          user={MockDetailedUser({
            isBanned: false,
            isDMCABanned: false,
            tosViolationCount: 0,
            dmcaViolationCount: 1
          })}
          onUnban={fn}
        />,
        []
      ).wrappedComponent
    );

    expect(
      wrapper
        .find({ "data-test-selector": TestSelector.DMCAViolationCountText })
        .hostNodes()
        .text()
    ).toBe("1 DMCA strikes");

    expect(
      wrapper
        .find({ "data-test-selector": TestSelector.TOSViolationCountText })
        .hostNodes()
        .text()
    ).toBe("0 TOS strikes");

    const decrementTOS = wrapper.find({ "data-test-selector": TestSelector.DecrementTOSCheckbox }).hostNodes();
    const decrementDMCA = wrapper.find({ "data-test-selector": TestSelector.DecrementDMCACheckbox }).hostNodes();
    const unban = wrapper.find({ "data-test-selector": TestSelector.UnbanButton }).hostNodes();
    expect(decrementTOS).toHaveLength(0);
    expect(decrementDMCA).toHaveLength(1);
    expect(unban).toHaveLength(1);

    // Checking the input this way since simulate doesn't work
    const input = decrementDMCA.find("input").getDOMNode() as HTMLInputElement;
    input.checked = true;
    decrementDMCA.find("input").simulate("change", { target: { checked: true } });

    unban.simulate("click");

    expect(fn.mock.calls).toHaveLength(0);

    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");

    expect(fn.mock.calls).toHaveLength(1);
  });
});
