import * as React from "react";

import { WithConfirmationModal } from "aegis/functionality/components/confirmation-modal";
import { DetailedUser } from "aegis/models";

import {
  AlignItems,
  Button,
  ButtonType,
  CheckBox,
  CoreText,
  Display,
  FlexDirection,
  FlexWrap,
  Icon,
  JustifyContent,
  Layout,
  Select,
  SVGAsset,
  Tooltip,
  TooltipDirection
} from "twitch-core-ui";

import "./styles.scss";

type Props = {
  user: DetailedUser;
  onUnban: (decrementCount: number, decrementTOS: boolean, decrementDMCA: boolean) => void;
};

export enum TestSelector {
  TOSViolationCountText = "account-info-strike-details--tos-violation-text",
  DMCAViolationCountText = "account-info-strike-details--dmca-violation-text",
  UnbanButton = "account-info-strike-details--unban-button",
  DecrementTOSCheckbox = "account-info-strike-details--decrement-tos-checkbox",
  DecrementDMCACheckbox = "account-info-strike-details--decrement-dmca-checkbox",
  DecrementCountSelect = "account-info-strike-details--decrement-select"
}

export const StrikeDetailsComponent: React.SFC<Props> = ({ user, onUnban }) => {
  return (
    <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
      <CoreText bold>Strike Counts</CoreText>
      <CoreText data-test-selector={TestSelector.TOSViolationCountText}>{user.tosViolationCount} TOS strikes</CoreText>
      <CoreText data-test-selector={TestSelector.DMCAViolationCountText}>
        {user.dmcaViolationCount} DMCA strikes
      </CoreText>

      {(user.tosViolationCount > 0 || user.dmcaViolationCount > 0) && <UnbanComponent user={user} onUnban={onUnban} />}
    </Layout>
  );
};

const UnbanComponent: React.SFC<Props> = ({ user, onUnban }) => {
  const [decrementCount, setDecrementCount] = React.useState(1);
  const [decrementTOS, setDecrementTOS] = React.useState(false);
  const [decrementDMCA, setDecrementDMCA] = React.useState(false);

  const count = decrementDMCA || decrementTOS ? decrementCount : 0;
  let messageBody = `Are you sure you want unban ${user.login}?`;
  if (count > 0) {
    const types: string[] = [];
    if (decrementTOS) {
      types.push("TOS");
    }
    if (decrementDMCA) {
      types.push("DMCA");
    }

    messageBody += ` This will also decrement ${count} from their ${types.join(" and ")} counts.`;
  }

  return (
    <WithConfirmationModal
      title={`Unban user ${user.login}`}
      body={messageBody}
      confirmLabelOverride="Unban"
      onConfirm={() => {
        onUnban(count, decrementTOS, decrementDMCA);
      }}
    >
      {confirm => {
        return (
          <Layout padding={{ top: 1 }}>
            <CoreText bold>Unban User</CoreText>

            <CoreText>Decrement Strikes (optional)</CoreText>
            <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
              <Layout display={Display.Flex} justifyContent={JustifyContent.Start} alignItems={AlignItems.Center}>
                {user.tosViolationCount > 0 && (
                  <CheckBox
                    data-test-selector={TestSelector.DecrementTOSCheckbox}
                    label="TOS"
                    checked={decrementTOS}
                    onChange={e => setDecrementTOS(e.currentTarget.checked)}
                  />
                )}
                <Layout padding={{ left: 1 }} />
                {user.dmcaViolationCount > 0 && (
                  <CheckBox
                    data-test-selector={TestSelector.DecrementDMCACheckbox}
                    label="DMCA"
                    checked={decrementDMCA}
                    onChange={e => setDecrementDMCA(e.currentTarget.checked)}
                  />
                )}
              </Layout>
              {(decrementDMCA || decrementTOS) && (
                <Layout display={Display.Flex} justifyContent={JustifyContent.Between} alignItems={AlignItems.Center}>
                  <Layout>Decrement</Layout>
                  <Layout className="strike_count_option">
                    <Select
                      data-test-selector={TestSelector.DecrementCountSelect}
                      defaultValue="1"
                      onChange={e => {
                        setDecrementCount(parseInt(e.currentTarget.value, 10));
                      }}
                    >
                      <option value={"1"}>1</option>
                      <option value={"2"}>2</option>
                      <option value={"3"}>3</option>
                    </Select>
                  </Layout>
                </Layout>
              )}
            </Layout>

            <Layout
              padding={{ y: 1 }}
              fullWidth
              display={Display.Flex}
              flexWrap={FlexWrap.NoWrap}
              justifyContent={JustifyContent.Between}
              alignItems={AlignItems.Center}
            >
              <Button
                data-track-click="user-account-info-unban"
                data-test-selector={TestSelector.UnbanButton}
                type={ButtonType.Hollow}
                disabled={!user.isBanned && !user.isDMCABanned && !decrementTOS && !decrementDMCA}
                onClick={confirm}
              >
                {user.isBanned || user.isDMCABanned ? "Unban" : "Decrement Strikes"}
              </Button>

              <Tooltip
                direction={TooltipDirection.Left}
                label="This will remove the suspension on this account regardless of type"
              >
                <Icon asset={SVGAsset.QuestionMark} />
              </Tooltip>
            </Layout>
          </Layout>
        );
      }}
    </WithConfirmationModal>
  );
};
