import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { AutoResolve as AutoResolveModel, PageInfo } from "aegis/models";
import { MockAutoResolve, MockMe, MockPageInfo, ToEdges } from "aegis/models/mocks";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { AutoResolve } from "../auto-resolve";
import { ActiveAutoResolveComponent, Props, TestSelectors } from "./component";

const newAutoResolveResponse = (autoResolves: AutoResolveModel[], pageInfo: PageInfo, total: number): Props => {
  return {
    activeAutoResolves: {
      edges: ToEdges(autoResolves),
      pageInfo: pageInfo,
      total: total
    },
    me: MockMe()
  };
};

Enzyme.configure({ adapter: new Adapter() });
describe("Active Auto Resolve", () => {
  it("renders with no auto resolves", () => {
    const props = newAutoResolveResponse([], MockPageInfo(), 0);
    const wrapper = Enzyme.shallow(<ActiveAutoResolveComponent {...props} />);

    expect(wrapper).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.ShowMore }).length).toBe(0);
  });

  it("renders with some auto resolves", () => {
    const props = newAutoResolveResponse([MockAutoResolve(), MockAutoResolve()], MockPageInfo(), 2);
    const wrapper = Enzyme.shallow(<ActiveAutoResolveComponent {...props} />);

    expect(wrapper).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.ShowMore }).length).toBe(0);
  });

  it("renders with other auto resolves", () => {
    const props = newAutoResolveResponse(
      [MockAutoResolve({ id: "not me" }), MockAutoResolve({ id: "not me" })],
      MockPageInfo(),
      2
    );
    const wrapper = Enzyme.shallow(<ActiveAutoResolveComponent {...props} />);

    expect(wrapper).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.ShowMore }).length).toBe(0);
  });

  it("renders with too many auto resolves", () => {
    const total = 10;
    const mocks: AutoResolveModel[] = [];
    for (var i = 0; i < total; i++) {
      mocks.push(MockAutoResolve({ id: `${i}` }));
    }
    const props = newAutoResolveResponse(mocks, MockPageInfo(), mocks.length);
    const wrapper = Enzyme.mount(wrapWithGQLProvider(<ActiveAutoResolveComponent {...props} />, []).wrappedComponent);

    expect(wrapper).toBeTruthy();

    const showMoreButton = wrapper.find({ "data-test-selector": TestSelectors.ShowMore }).hostNodes();
    expect(showMoreButton).toHaveLength(1);
    expect(wrapper.find(AutoResolve)).toHaveLength(5);

    showMoreButton.simulate("click");

    expect(wrapper.find(AutoResolve)).toHaveLength(10);
  });
});
