import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";

import { AutoResolve } from "aegis/features/auto-resolve";
import { CollapsibleCard } from "aegis/functionality/components/collapsible";
import { PanelTitle } from "aegis/functionality/components/panel-title";
import { AutoResolve as AutoResolveModel, Me, PagedResponse } from "aegis/models";
import {
  AlignItems,
  Background,
  Button,
  ButtonType,
  CoreText,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  StyledLayout
} from "twitch-core-ui";

import "./styles.scss";

export interface Props {
  activeAutoResolves: PagedResponse<AutoResolveModel> & {
    total: number;
  };
  me: Me;
}

export enum TestSelectors {
  ShowMore = "auto-resolve-card--show-more"
}

@observer
export class ActiveAutoResolveComponent extends React.Component<Props> {
  @observable private showMore: boolean = false;

  public render() {
    const { activeAutoResolves, me } = this.props;

    const total = activeAutoResolves.edges.length;

    const buttonText = !this.showMore ? "Show More" : "Show Less";
    const showMoreButton = total > 5 && (
      <Button
        data-track-click="active-auto-resolve-card-show-more"
        onClick={this.toggleShowMore}
        fullWidth
        type={ButtonType.Text}
        data-test-selector={TestSelectors.ShowMore}
      >
        {buttonText}
      </Button>
    );

    const limit = this.showMore ? total : Math.min(5, total);

    const nodes = activeAutoResolves.edges.slice(0, limit).map(edge => edge.node);

    const myAutoResolves = nodes.filter(node => node.createdBy && node.createdBy.id === me.id);
    const otherAutoResolves = nodes.filter(node => !myAutoResolves.includes(node));

    return (
      <StyledLayout>
        <StyledLayout
          display={Display.Flex}
          justifyContent={JustifyContent.Center}
          flexDirection={FlexDirection.Column}
          alignItems={AlignItems.Center}
          border
          elevation={2}
          className="auto-resolve-card report-side-card"
        >
          <CollapsibleCard title={<PanelTitle title="Active autoresolves" length={activeAutoResolves.total} />}>
            <DisplayAutoResolves title="Rules by me" autoresolves={myAutoResolves} />
            <DisplayAutoResolves title="Rules by others" autoresolves={otherAutoResolves} showCreatedBy />
            {showMoreButton}
          </CollapsibleCard>
        </StyledLayout>
      </StyledLayout>
    );
  }

  @action
  private toggleShowMore = () => {
    this.showMore = !this.showMore;
  };
}

const DisplayAutoResolves: React.SFC<{ title: string; autoresolves: AutoResolveModel[]; showCreatedBy?: boolean }> = ({
  title,
  autoresolves,
  showCreatedBy
}) => {
  const total = autoresolves.length;
  if (total === 0) {
    return null;
  }

  const autoResolves = autoresolves.map((a, i) => (
    <AutoResolve key={a.id} autoResolve={a} showCreatedBy={showCreatedBy} borderBottom={i + 1 !== total} />
  ));
  return (
    <Layout fullWidth>
      <StyledLayout borderBottom borderTop elevation={1} padding={{ left: 1 }} background={Background.Alt}>
        <CoreText bold>{title}</CoreText>
      </StyledLayout>
      <Layout>{autoResolves}</Layout>
    </Layout>
  );
};
