import * as React from "react";

import gql from "graphql-tag";

import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { AutoResolve, Me, PagedResponse, PollingInterval } from "aegis/models";
import { Color, CoreText, LoadingSpinner } from "twitch-core-ui";
import { ActiveAutoResolveComponent } from "./component";

// Only pull up to 1000 auto resolves since admins are expected to track these manually
const recordsToFetch = 1000;

export const ActiveAutoResolvesQuery = gql`
  query activeAutoResolves {
    activeAutoResolves(first: ${recordsToFetch}) {
      edges {
        node {
          ...AutoResolveFields
        }
      }
      total
    }

    me {
      id
    }
  }
  ${Fragments.AutoResolveFields}
`;

interface Response {
  activeAutoResolves: PagedResponse<AutoResolve> & {
    total: number;
  };
  me: Me;
}

class ActiveAutoResolveCardQuery extends Query<Response> {}

export const ActiveAutoResolveCard = () => (
  <ActiveAutoResolveCardQuery
    displayName="ActiveAutoResolveCard"
    pollInterval={PollingInterval.Fast}
    query={ActiveAutoResolvesQuery}
  >
    {({ loading, error, data }) => {
      // Only show loading spinner if it's first load (loading + no data)
      // to keep a consistent view while loading new data
      if (loading && (!data || !data.activeAutoResolves)) {
        return <LoadingSpinner />;
      }

      if (error) {
        console.error(error);
        return (
          <CoreText color={Color.Error}>An error occurred while loading active auto resolves: {error.message}</CoreText>
        );
      }

      if (!data) {
        throw new Error("No data returned from server");
      }

      if (!data.activeAutoResolves || !data.activeAutoResolves.edges) {
        console.error("data did not have active autoresolve.");
        return <CoreText color={Color.Error}> Something is wrong. No Active auto resolves found</CoreText>;
      }

      return <ActiveAutoResolveComponent activeAutoResolves={data.activeAutoResolves} me={data.me} />;
    }}
  </ActiveAutoResolveCardQuery>
);
