import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { AltUserInfo } from "aegis/models";
import { MockAltUserInfo, MockUser } from "aegis/models/mocks";

import { AltLookUpStore } from "aegis/stores/altLookUpStore";
import { AltLookupTableComponent, TestSelectors } from "./component";
import { AltLookupTableRow } from "./row";

Enzyme.configure({ adapter: new Adapter() });

describe("Alt lookup table", () => {
  let altLookUpStore: AltLookUpStore;
  const altUsers: AltUserInfo[] = [
    MockAltUserInfo({ user: MockUser({ id: "1" }) }),
    MockAltUserInfo({ user: MockUser({ id: "22" }) }),
    MockAltUserInfo({ user: MockUser({ id: "333" }) })
  ];
  beforeEach(() => {
    altLookUpStore = new AltLookUpStore();
    altLookUpStore.setAltUsers(altUsers);
  });

  it("renders", () => {
    const wrapper = Enzyme.shallow(<AltLookupTableComponent altUsers={altUsers} altLookUpStore={altLookUpStore} />);

    expect(wrapper.dive().find(AltLookupTableRow)).toHaveLength(altUsers.length);
  });

  it("toggles all users", () => {
    const wrapper = Enzyme.mount(<AltLookupTableComponent altUsers={altUsers} altLookUpStore={altLookUpStore} />);

    wrapper
      .find({ "data-test-selector": TestSelectors.ToggleAllBox })
      .first()
      .find("input")
      .simulate("change");

    expect(altLookUpStore.allSelected).toBe(true);
    expect(altLookUpStore.refs.every(ref => ref.checked)).toBe(true);

    wrapper
      .find({ "data-test-selector": TestSelectors.ToggleAllBox })
      .first()
      .find("input")
      .simulate("change");

    expect(altLookUpStore.allSelected).toBe(false);
    expect(altLookUpStore.refs.every(ref => !ref.checked)).toBe(true);
  });

  it("one user can be selected", () => {
    const wrapper = Enzyme.mount(<AltLookupTableComponent altUsers={altUsers} altLookUpStore={altLookUpStore} />);
    const input = wrapper
      .find(AltLookupTableRow)
      .at(1)
      .find("input")
      .getDOMNode() as HTMLInputElement;

    // simulating an input's 'checked' attribute change for refs
    input.checked = true;
    // simulating change to invoke onChange props
    wrapper
      .find(AltLookupTableRow)
      .at(1)
      .find("input")
      .simulate("change", { target: { checked: true } });

    expect(altLookUpStore.refs[1].checked).toBe(true);
    expect(altLookUpStore.selectedRows).toMatchObject([false, true, false]);
  });

  it("one user can be deselected when all toggled", () => {
    const wrapper = Enzyme.mount(<AltLookupTableComponent altUsers={altUsers} altLookUpStore={altLookUpStore} />);
    wrapper
      .find({ "data-test-selector": TestSelectors.ToggleAllBox })
      .find("input")
      .simulate("change");

    expect(altLookUpStore.allSelected).toBe(true);
    expect(altLookUpStore.refs.every(ref => ref.checked)).toBe(true);

    const input = wrapper
      .find(AltLookupTableRow)
      .at(1)
      .find("input")
      .getDOMNode() as HTMLInputElement;

    // simulating an input's 'checked' attribute change for refs
    input.checked = false;
    wrapper
      .find(AltLookupTableRow)
      .at(1)
      .find("input")
      .simulate("change");

    expect(altLookUpStore.refs.every(ref => ref.checked)).toBe(false);
    expect(altLookUpStore.refs[1].checked).toBe(false);
    expect(altLookUpStore.selectedRows).toMatchObject([true, false, true]);
  });
});
