import * as React from "react";

import { inject, observer } from "mobx-react";

import { copyTextToClipboard } from "aegis/functionality/utils/contrib/twilight/copy-text-to-clipboard";
import { getKeysOfSelection } from "aegis/functionality/utils/get-keys";
import { AltUserInfo } from "aegis/models";
import { AltLookUpStore } from "aegis/stores/altLookUpStore";
import {
  CheckBox,
  Layout,
  Table,
  TableBody,
  TableHeader,
  TableHeading,
  TextAlign,
  VerticalAlign
} from "twitch-core-ui";
import { AltLookupTableRow } from "./row";

import "./styles.scss";

export interface Props {
  altUsers: AltUserInfo[];
  altLookUpStore?: AltLookUpStore;
}

export enum TestSelectors {
  ToggleAllBox = "alt-look-up--toggle-all-box"
}

@inject("altLookUpStore")
@observer
export class AltLookupTableComponent extends React.Component<Props> {
  constructor(props: Props) {
    super(props);

    this.props.altLookUpStore!.setAltUsers(this.props.altUsers);
    this.props.altLookUpStore!.setAllSelected(false);
  }

  public render() {
    const { altUsers, altLookUpStore } = this.props;
    return (
      <div className="alt-lookup-table" onCopy={this.onCopy}>
        <Table>
          <TableHeader data-key={-1} data-alt-look-up-row="alt-look-up-row">
            <Layout className="tw-table-heading" textAlign={TextAlign.Center} verticalAlign={VerticalAlign.Middle}>
              <Layout padding={1}>
                <CheckBox
                  data-track-click="alt-lookup-toggle-all"
                  label={""}
                  refDelegate={altLookUpStore!.checkAllRefDelegate}
                  onChange={altLookUpStore!.toggleAll()}
                  data-test-selector={TestSelectors.ToggleAllBox}
                />
              </Layout>
            </Layout>
            <TableHeading label="Login" />
            <TableHeading label="Type" />
            <TableHeading label="Matched User" />
            <TableHeading label="Created On" />
            <TableHeading label="Actions" />
          </TableHeader>
          <TableBody>
            {altUsers.map((altUser: AltUserInfo, i: number) => {
              return (
                <AltLookupTableRow
                  altUser={altUser}
                  key={`alt-user-${altUser.user.id}-${i}`}
                  index={i}
                  refDelegate={altLookUpStore!.refDelegates[i]}
                  onCheck={altLookUpStore!.updateSelected()}
                />
              );
            })}
          </TableBody>
        </Table>
      </div>
    );
  }

  private onCopy = (event: React.ClipboardEvent<HTMLElement>) => {
    const { altUsers } = this.props;
    const selection = window.getSelection();
    if (selection === null) {
      return;
    }
    const { first, last } = getKeysOfSelection(selection, "data-alt-look-up-row");
    if (first === last) {
      return;
    }

    let start = first;
    let header = "";
    // Check if table header was selected. TableHeader has attribute data-key=-1
    if (first < 0 && last >= 0) {
      start = 0;
      const loginSpacer = " ".repeat(48 - "Login".length);
      const typeSpacer = " ".repeat(15 - "Type".length);
      const matchedUserSpacer = " ".repeat(48 - "Matched User".length);
      header = `Login${loginSpacer}Type${typeSpacer}Matched User${matchedUserSpacer}CreatedOn\n`;
    }

    // Prevent race condition of initial copy propogating after this function
    event.preventDefault();

    // Stores selected range before it is wiped by copyTextToClipboard
    const selectedRange = selection.getRangeAt(0);

    const selectedAltUserRows = altUsers.slice(start, last ? last : altUsers.length);
    const result = selectedAltUserRows
      .map(altUserInfo => {
        const { user, type, matched } = altUserInfo;

        const altBanStatus = user.isBanned ? "[TOS] " : user.isDMCABanned ? "[DMCA] " : "";
        const altUser = `${altBanStatus}${user.login} (${user.id})`;
        const altUserSpacer = " ".repeat(48 - altUser.length);

        const typeSpacer = " ".repeat(15 - type.length);

        const matchedBanStatus = matched.isBanned ? "[TOS] " : matched.isDMCABanned ? "[DMCA] " : "";
        const matchedUser = `${matchedBanStatus}${matched.login} (${matched.id})`;
        const matchedUserSpacer = " ".repeat(48 - matchedUser.length);

        return `${altUser}${altUserSpacer}${type}${typeSpacer}${matchedUser}${matchedUserSpacer}${user.createdAt}`;
      })
      .join("\n");

    copyTextToClipboard(header + result);

    // Restores selected area, unselected by copyTextToClipboard
    selection.removeAllRanges();
    selection.addRange(selectedRange);
  };
}
