import * as React from "react";

import gql from "graphql-tag";
import { Color, CoreText, LoadingSpinner } from "twitch-core-ui";

import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { AltUserInfo } from "aegis/models";
import { AltLookupTableComponent } from "./component";

export const ALT_USER_QUERY = gql`
  query altUsers($id: ID!) {
    altUsers(id: $id) {
      type
      user {
        ...UserFields
      }
      matched {
        ...UserFields
      }
    }
  }
  ${Fragments.UserFields}
`;

interface Data {
  altUsers: AltUserInfo[];
}

interface Variables {
  id: string;
}

class AllAltsQuery extends Query<Data, Variables> {}

interface Props {
  id?: string;
}

export const AltLookupTable: React.SFC<Props> = ({ id }) => {
  if (id === undefined) {
    return null;
  }
  return (
    <AllAltsQuery displayName="AltLookupTable" query={ALT_USER_QUERY} variables={{ id: id }}>
      {({ loading, error, data }) => {
        if (loading) {
          return <LoadingSpinner />;
        }

        if (error) {
          console.error("AllAltsQuery errored:", error);
          return <CoreText color={Color.Error}>An error occurred while loading alt users: {error.message}</CoreText>;
        }

        // handle lingering null from an errored query: https://github.com/graph-gophers/graphql-go/issues/275
        if (!data || data.altUsers === null) {
          console.error("AllAltsQuery: null data");
          return null;
        }

        return <AltLookupTableComponent altUsers={data!.altUsers} />;
      }}
    </AllAltsQuery>
  );
};
