import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { MockAltUserInfo, MockUser } from "aegis/models/mocks";
import { AltLookUpStore } from "aegis/stores/altLookUpStore";
import { LinkToChatActivity, TestSelectors } from "./component";

Enzyme.configure({ adapter: new Adapter() });

describe("Alt lookup table row", () => {
  let altLookUpStore: AltLookUpStore;
  beforeEach(() => {
    altLookUpStore = new AltLookUpStore();
    altLookUpStore.setAltUsers([
      MockAltUserInfo({ user: MockUser({ id: "1" }) }),
      MockAltUserInfo({ user: MockUser({ id: "22" }) }),
      MockAltUserInfo({ user: MockUser({ id: "333" }) })
    ]);
    altLookUpStore.refDelegates.forEach(refDelegate => {
      refDelegate({ checked: false } as HTMLInputElement);
    });
  });

  it("renders nothing when no alt users selectable", () => {
    const wrapper = Enzyme.shallow(<LinkToChatActivity altLookUpStore={new AltLookUpStore()} />);
    const coreText = wrapper.dive().find({ "data-test-selector": TestSelectors.SelectedCount });
    const buttonLink = wrapper.dive().find({ "data-test-selector": TestSelectors.ButtonLink });
    expect(coreText).toHaveLength(0);
    expect(buttonLink).toHaveLength(0);
  });

  it("links when all users selected", () => {
    altLookUpStore.toggleAll()();
    const wrapper = Enzyme.shallow(<LinkToChatActivity altLookUpStore={altLookUpStore} />);
    const coreText = wrapper.dive().find({ "data-test-selector": TestSelectors.SelectedCount });
    const buttonLink = wrapper.dive().find({ "data-test-selector": TestSelectors.ButtonLink });

    expect(coreText.childAt(0).text()).toBe("3 users selected.");
    expect(buttonLink).toHaveLength(1);
    expect(buttonLink.prop("linkTo")).toBe("/chatactivity?users=1,22,333");
  });

  it("links when only 1 selected", () => {
    altLookUpStore.toggleRow(1);
    const wrapper = Enzyme.shallow(<LinkToChatActivity altLookUpStore={altLookUpStore} />);

    const coreText = wrapper.dive().find({ "data-test-selector": TestSelectors.SelectedCount });
    const buttonLink = wrapper.dive().find({ "data-test-selector": TestSelectors.ButtonLink });

    expect(coreText.childAt(0).text()).toBe("1 users selected.");
    expect(buttonLink).toHaveLength(1);
    expect(buttonLink.prop("linkTo")).toBe("/chatactivity?users=22");
  });

  it("links disabled when 0 selected", () => {
    const wrapper = Enzyme.shallow(<LinkToChatActivity altLookUpStore={altLookUpStore} />);

    const coreText = wrapper.dive().find({ "data-test-selector": TestSelectors.SelectedCount });
    const buttonLink = wrapper.dive().find({ "data-test-selector": TestSelectors.ButtonLink });

    expect(coreText.childAt(0).text()).toBe("0 users selected.");
    expect(buttonLink).toHaveLength(1);
    expect(buttonLink.prop("disabled")).toBe(true);
  });

  it("links when only all selected, 1 deselected", () => {
    altLookUpStore.toggleAll()();
    altLookUpStore.toggleRow(1);
    const wrapper = Enzyme.shallow(<LinkToChatActivity altLookUpStore={altLookUpStore} />);

    const coreText = wrapper.dive().find({ "data-test-selector": TestSelectors.SelectedCount });
    const buttonLink = wrapper.dive().find({ "data-test-selector": TestSelectors.ButtonLink });

    expect(coreText.childAt(0).text()).toBe("2 users selected.");
    expect(buttonLink).toHaveLength(1);
    expect(buttonLink.prop("linkTo")).toBe("/chatactivity?users=1,333");
  });
});
